/*
 * ADC.c
 *
 * Created: 2013-07-28 16:37:48
 *  Author: tmf
 */

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <stddef.h>
#include <avr/interrupt.h>
#include <string.h>
#include <stdio.h>
#include "ADC.h"
#include "SPI.h"
#include "AT45IO.h"
#include "AT45CMD.h"
#include "RTC.h"

__uint24 DF_Page;          //Nr zapisywanej strony
uint16_t DF_Offset;        //Offset w buforze pamici DataFLASH
uint8_t DF_BufPage;        //Nr aktualnie uywanej strony bufora

void DF_PrepareToBufWrite() //Przygotuj odpowiedni stron bufora do zapisu
{
	DataFLASH_CSEnable(false);       //Zakocz poprzednie polecenie
	if(DF_BufPage == 0) DataFLASH_SendCmdAddr(AT45DBX_CMDC_WR_BUF1, DF_Offset, CS_Low); //Ustaw pami w tryb zapisu bufora nr 1
			else DataFLASH_SendCmdAddr(AT45DBX_CMDC_WR_BUF2, DF_Offset, CS_Low); //Ustaw pami w tryb zapisu bufora nr 2
	if(DF_Offset == 0)
		for(uint16_t i=0; i < DataFLASH_PageSize; i++) SPI_RW_Byte(0xff); //Kasujemy bufor strony - dziki temu strona nie zawiera mieci
}

void DF_FinishBufWrite()
{
	DataFLASH_CSEnable(false);       //Deaktywujemy pami w celu oszczdzania energii
	if(DF_Offset == DataFLASH_PageSize)  //Bufor peny, czas zapisa dane
	{
		DF_Offset=0;                     //Zerujemy wskanik zapisu w buforze
		if(DF_BufPage == 0) DataFLASH_SendCmdAddr(AT45DBX_CMDB_PR_BUF1_TO_PAGE_ER, DF_Page, CS_High);
			else DataFLASH_SendCmdAddr(AT45DBX_CMDB_PR_BUF2_TO_PAGE_ER, DF_Page, CS_High);
		DF_Page+=DataFLASH_PageSize; //Zapisujemy kolejn stron
		DF_BufPage^=0x01;            //Zmie uywany bufor
	}
}

void DF_SynchBuffers()     //Natychmiast zapisuje stan bufora DataFLASH
{
	if(DF_Offset == 0) return;
	DataFLASH_CSEnable(false);       //Zakocz poprzednie polecenie
	DataFLASH_WaitForBusy();  //Zaczekaj na koniec operacji

	if(DF_BufPage == 0) DataFLASH_SendCmdAddr(AT45DBX_CMDB_PR_BUF1_TO_PAGE_ER, DF_Page, CS_High);
		else DataFLASH_SendCmdAddr(AT45DBX_CMDB_PR_BUF2_TO_PAGE_ER, DF_Page, CS_High);
	DataFLASH_WaitForBusy();  //Zaczekaj na koniec operacji
}

void SendTimestamp()
{
	uint32_t Timestamp=RTC32_get_counter(); //Pobierz aktualny czas
	SPI_RW_Byte(Timestamp & 0xff);
	SPI_RW_Byte((Timestamp >> 8) & 0xff);
	SPI_RW_Byte((Timestamp >> 16) & 0xff);
	SPI_RW_Byte((Timestamp >> 24) & 0xff);
	DF_Offset+=4;
}

void SendVoltage(int16_t ADCtemp)
{
	ADCtemp=ADCtemp*1000L/2048;  //Przelicz temperatur z LM35
	SPI_RW_Byte(ADCtemp & 0xff); //Wylij modszy bajt temperatury
	SPI_RW_Byte(ADCtemp >> 8);   //Wylij starszy bajt temperatury
	DF_Offset+=2;
}

ISR(ADCA_CH0_vect)
{
	PORTR_OUTCLR=PIN0_bm;          //Wcz diod sygnalizujc akwizycj
	DF_PrepareToBufWrite();        //Przygotuj pami na zapis bufora - pocztek zapisywanych danych
	SendTimestamp();
	SendVoltage(ADCA_CH0_RES);
}

ISR(ADCA_CH1_vect)
{
	SendVoltage(ADCA_CH2_RES);
}

ISR(ADCA_CH2_vect)
{
	SendVoltage(ADCA_CH1_RES);
}

ISR(ADCA_CH3_vect)
{
	SendVoltage(ADCA_CH3_RES);
	DF_FinishBufWrite();            //Zakocz biec porcj zapisywanych danych
}

uint8_t ReadCalibrationByte(uint8_t index)
{
	uint8_t result;

	NVM_CMD=NVM_CMD_READ_CALIB_ROW_gc; //Odczytaj sygnatur produkcyjn
	result=pgm_read_byte(index);

	NVM_CMD=NVM_CMD_NO_OPERATION_gc;   //Przywr normalne dziaanie NVM
	return result;
}

void ADC_CH_Init(ADC_CH_t *adcch, register8_t muxpos)
{
	adcch->CTRL=ADC_CH_INPUTMODE_SINGLEENDED_gc;  //Tryb pojedynczego wejcia ze znakiem
	adcch->MUXCTRL=muxpos;                        //Pin wejcia dodatniego
	adcch->INTCTRL=ADC_CH_INTMODE_COMPLETE_gc | ADC_CH_INTLVL_LO_gc;
}

ISR(TCC0_CCA_vect)
{
	PORTR_OUTSET=PIN0_bm;
}

void Timer_Init()
{
	TCC0.CTRLB=TC_WGMODE_NORMAL_gc | TC0_CCAEN_bm;       //Zwyky tryb pracy timera i odblokowany kana CCA
	TCC0.PER=F_CPU/256;                    //Odliczamy 1 sekund
	TCC0.CCA=TCC0.PER/10;                  //Zdarzenie CCA z opnieniem ok. 100 ms
	TCC0.INTCTRLB=TC_CCAINTLVL_LO_gc;
	EVSYS_CH0MUX=EVSYS_CHMUX_TCC0_OVF_gc;  //Routowane do kanau zdarze nr 0
	PORTR.DIRSET=PIN0_bm;                  //Pin sterujcy LED0
	PORTR.OUTSET=PIN0_bm;                  //Wycz diod
}

void DataFlash_Init()
{
	uint8_t Bufor[DataFLASH_PageSize];
	DataLogger_t *dl;

	DF_Page=0; DF_Offset=0;
	DF_BufPage=0;

	do
	{
		DataFLASH_ReadSeq(DF_Page, Bufor, DataFLASH_PageSize); //Odczytaj stron pamici
		while(DF_Offset < DataFLASH_PageSize)
		{
			dl=(DataLogger_t*)&Bufor[DF_Offset];
			if(dl->Timestamp == -1)
			{
				DataFLASH_SendCmdAddr(AT45DBX_CMDB_XFR_PAGE_TO_BUF1, DF_Page, CS_High); //Wpisz do bufora 1 ostatni stron pamici - w razie doczania danych
				DataFLASH_WaitForBusy();  //Zaczekaj na koniec operacji
				return;                   //Znalelimy koniec danych, wic wracamy
			}
			DF_Offset+=sizeof(DataLogger_t);
		}
		DF_Offset%=DataFLASH_PageSize;
		DF_Page+=DataFLASH_PageSize;
	} while(1);
}

void ADC_Init()
{
	DataFlash_Init();
	ADCA.CTRLA=ADC_ENABLE_bm;
	ADCA.CTRLB=ADC_CONMODE_bm; //Rozdzielczo 12 bitw, tryb ze znakiem
	ADCA.REFCTRL=ADC_REFSEL_INT1V_gc | ADC_BANDGAP_bm; //Referencja 1V
	ADCA.EVCTRL=ADC_SWEEP_0123_gc | ADC_EVSEL_0123_gc | ADC_EVACT_SWEEP_gc; //Wyzwalanie kanaw 0, 1, 2 i 3 przez EVCH0
	ADCA.PRESCALER=ADC_PRESCALER_DIV16_gc;     //CLKADC=1 MHz
	ADCA.CALL=ReadCalibrationByte(offsetof(NVM_PROD_SIGNATURES_t, ADCACAL0));
	ADCA.CALH=ReadCalibrationByte(offsetof(NVM_PROD_SIGNATURES_t, ADCACAL1));

	ADC_CH_Init(&ADCA.CH0, ADC_CH_MUXPOS_PIN0_gc);   //Zainicjuj poszczeglne kanay ADC
	ADC_CH_Init(&ADCA.CH1, ADC_CH_MUXPOS_PIN1_gc);
	ADC_CH_Init(&ADCA.CH2, ADC_CH_MUXPOS_PIN2_gc);
	ADC_CH_Init(&ADCA.CH3, ADC_CH_MUXPOS_PIN3_gc);
	Timer_Init();
}
