/*
 * OdbiornikIR_NEC.c
 *
 * Created: 2013-01-04 16:00:55
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdbool.h>
#include <stdlib.h>
#include <avr/pgmspace.h>
#include "usart.h"
#include "RingBuffer.h"

#define IR_NEC_PULSE        560  //Czas w us podstawowego interwau
#define IR_NEC_TOLERANCE    100  //Tolerancja czasu w us
#define IR_NEC_TRAILER      16   // Start ramki - 16 interwaw
#define IR_NEC_ZERO         2    //Czas trwania bitu 0
#define IR_NEC_ONE          4    //Czas trwania bitu 1
#define IR_BITSNO           32   //Liczba bitw kodu
#define IR_NEC_RPTLASTCMD   -1   //Kod powtrzenia ostatniego polecenia

static inline uint16_t IR_CalcTime(uint16_t time)
{
	return time*(F_CPU/1000000UL)/4;   //Przelicz czas w mikrosekundach na tyknicia timera
	                                   //Przy timerze taktowanym CLKPER/4
}

enum IR_NEC_States {IR_NEC_Nothing, IR_NEC_Trailer, IR_NEC_FirstBit, IR_NEC_Receiving};
enum IR_NEC_States IR_State;           //Stan maszyny stanw

volatile uint8_t  IR_recpos;     //Nr aktualnie odbieranego bitu

CircBuffer IR_CMD_Buffer; //Instancja bufora koowego przechowujcego polecenia

ISR(TCD0_OVF_vect)
{
    if((IR_State==IR_NEC_Receiving) && (IR_recpos==0))
		cb_Add(&IR_CMD_Buffer, IR_NEC_RPTLASTCMD);
	IR_State=IR_NEC_Nothing;      //Bd transmisji lub powtrzenie - restart maszyny stanw
}

ISR(TCD0_CCA_vect)
{
    static uint32_t IR_RecCmd;   //Odebrana komenda z pilota
	
	TCD0_CNT=0;             //Co odebralimy wic zerujemy licznik
	uint16_t cca=TCD0_CCA;
	uint8_t flag=0;
	if(cca & 0x8000) flag=1;
	cca&=0x7fff;
	switch(IR_State)
	{
		case IR_NEC_Nothing:
		       if(flag==0)
		        {
		          IR_State=IR_NEC_Trailer;
		          IR_recpos=0;
				  IR_RecCmd=0;
	            }
			break;

		case IR_NEC_Trailer:
		       if((cca>IR_CalcTime((IR_NEC_PULSE-IR_NEC_TOLERANCE)*IR_NEC_TRAILER))  //Koniec odbioru trailera
		            && ((cca<IR_CalcTime((IR_NEC_PULSE+IR_NEC_TOLERANCE)*IR_NEC_TRAILER))))
			              IR_State=IR_NEC_FirstBit;
			break;

		case IR_NEC_FirstBit:
		        IR_State=IR_NEC_Receiving;
			break;

        case IR_NEC_Receiving:
		        if(flag==0)
		         {
				   IR_RecCmd<<=1;
				   if(cca>IR_CalcTime((IR_NEC_PULSE+IR_NEC_TOLERANCE)*IR_NEC_ZERO)) IR_RecCmd|=1;
			       ++IR_recpos;
			       if(IR_recpos==IR_BITSNO)
				   {
					   IR_State=IR_NEC_Nothing;
					   cb_Add(&IR_CMD_Buffer, IR_RecCmd);  //Dodaj odebrane polecenie do bufora
				   }					   
		         }
			break;
	}
}

void IR_init()
{
	PORTD_PIN2CTRL=PORT_OPC_PULLUP_gc | PORT_ISC_BOTHEDGES_gc;  //Pulup na pinie odbiornika IR, zbocza wywouj zdarzenia
	EVSYS_CH1MUX=EVSYS_CHMUX_PORTD_PIN2_gc;                     //Transmitowane przez kana 1
	TCD0.CTRLB=TC_WGMODE_NORMAL_gc | TC0_CCAEN_bm;              //Tryb pracy normalny
	TCD0.PER=IR_CalcTime((IR_NEC_PULSE+IR_NEC_TOLERANCE)*IR_NEC_TRAILER);         //Okres timera
	TCD0.CTRLD=TC_EVSEL_CH1_gc | TC_EVACT_CAPT_gc;              //Kana 1 jako capture
	TCD0.INTCTRLA=TC_OVFINTLVL_LO_gc;      //Wcz przerwanie nadmiaru
	TCD0.INTCTRLB=TC_CCAINTLVL_LO_gc;      //Wcz przerwanie CCA
	TCD0.CTRLA=TC_CLKSEL_DIV4_gc;
	PMIC_CTRL=PMIC_LOLVLEN_bm;             //Odblokuj przerwania niskiego poziomu
}

void USART_init()
{
	USARTC0.CTRLB=USART_TXEN_bm;         //Wcz nadajnik USART
	USARTC0.CTRLC=USART_CHSIZE_8BIT_gc;  //Ramka 8 bitw, bez parzystoci, 1 bit stopu
	usart_set_baudrate(&USARTC0, 9600, F_CPU);
	PORTC_DIRSET=PIN3_bm;                //Pin TxD musi by wyjciem
}

int main(void)
{
	char bufor[13];
	
	USART_init();
	IR_init();
	sei();
    while(1)
    {
		if(cb_IsEmpty(&IR_CMD_Buffer)==false)
		{
			CB_Element cmd=cb_Read(&IR_CMD_Buffer);
			if(cmd!=IR_NEC_RPTLASTCMD)
			{
				ultoa(cmd, bufor, 16);
			    USART_send(&USARTC0, bufor);
			} else USART_send_F(&USARTC0, PSTR("Powtorzenie"));
			USART_putchar(&USARTC0, '\n');
			USART_putchar(&USARTC0, '\r');
		}			
    }
}