/*
 * Matryca_multikolor.c
 *
 * Created: 2013-03-15 20:50:08
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <string.h>
#include <stdbool.h>
#include <avr/pgmspace.h>

#include "font5x8.h"         //Tablica czcionek 5x8

#define M_WITDTH   6         //Dugo wiersza matrycy w bajtach
#define M_HEIGHT   8         //Wysoko matrycy w wierszach
#define LED_REFRESHRATE 300  //Czstotliwo odwierzania matrycy
#define M_COLOR    8         //Liczba kolorw na kana

uint8_t Bufor[M_COLOR][M_HEIGHT][M_WITDTH+1]; //Miejsce na bufor i adres wiersza
	
void SPI_init()
{
	PORTC.OUTCLR=PIN1_bm;
	PORTC.DIRSET=PIN3_bm | PIN1_bm;         //Pin TxD i SCK jest wyjciem
	PORTC.PIN1CTRL=PORT_ISC_RISING_gc;      //SCK generuje zdarzenie na zboczu narastajcym
	EVSYS_CH1MUX=EVSYS_CHMUX_PORTC_PIN1_gc; //Transmitowane do kanau zdarze nr 1
	
	USARTC0.CTRLB=USART_TXEN_bm;       //Odblokuj MOSI
	USARTC0.CTRLC=USART_CMODE_MSPI_gc; //Wybierz tryb SPI, MSB najpierw,
	USARTC0.BAUDCTRLA=(F_CPU/((M_WITDTH+1)*M_HEIGHT*8UL*LED_REFRESHRATE*M_COLOR))-1; //Oblicz wymagany transfer w zalenoci od odwieania, wielkoci matrycy i liczby kolorw
	USARTC0.BAUDCTRLB=0;
}

void DMA_init()
{
	DMA.CH0.SRCADDR0=(uint16_t)Bufor & 0xff;
	DMA.CH0.SRCADDR1=((uint16_t)Bufor >> 8) & 0xff;
	DMA.CH0.SRCADDR2=0;
	DMA.CH0.DESTADDR0=((uint16_t)&USARTC0_DATA) & 0xff;
	DMA.CH0.DESTADDR1=((uint16_t)&USARTC0_DATA >> 8) & 0xff;
	DMA.CH0.DESTADDR2=0;
	DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTRELOAD_BURST_gc | DMA_CH_DESTDIR_FIXED_gc;
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_USARTC0_DRE_gc; //Pusty bufor USART wyzwala zdarzenie
	DMA.CH0.REPCNT=0;                     //Nieskoczona liczba transferw
	DMA.CH0.TRFCNT=sizeof(Bufor);         //Transferujemy jako blok ca tablic z danymi
	DMA.CH0.CTRLA=DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc | DMA_CH_REPEAT_bm;
	DMA.CH0.CTRLA|=DMA_CH_ENABLE_bm;      //Odblokuj transfer
	DMA.CTRL=DMA_ENABLE_bm;               //Odblokuj DMA
}

void TimerEv_init()
{
	TCC1.PER=(M_WITDTH+1)*8-1;              //Liczba bitw transmisji
	TCC1.CTRLA=TC_CLKSEL_EVCH1_gc;          //Timer taktowany jest z EVCH1, czyli impulsami taktujcymi transmisj SPI
	EVSYS_CH0MUX=EVSYS_CHMUX_TCC1_OVF_gc;   //Zdarzenie nadmiaru bdzie transmitowane do EVCH0
	PORTC.DIRSET=PIN7_bm;                   //PC7 jest wyjciem na ktrym bdzie EV_CH0
	PORTC.OUTCLR=PIN7_bm;
	PORTCFG_CLKEVOUT=PORTCFG_EVOUT_PC7_gc;  //Udostnij wyjcie z event system na PC7
	PORTCFG_EVOUTSEL=PORTCFG_EVOUTSEL_0_gc; //Na PC7 jest kana EV_CH0
}

void Bufor_init()
{
	uint8_t mask=0xfe;  //Maska aktywnego rzdu
	for(uint8_t h=0; h<M_HEIGHT; h++)
	{
	 for(uint8_t z=0; z<M_COLOR; z++) Bufor[z][h][M_WITDTH]=mask;
	 mask=mask<<1 | 1;
	}	 
}

bool OSC_wait_for_rdy(uint8_t clk)
{
	uint8_t czas=255;
	while ((!(OSC.STATUS & clk)) && (--czas)) // Czekaj na ustabilizowanie si generatora
	_delay_ms(1);
	return czas;   //false jeli generator nie wystartowa, true jeli jest ok
}

bool RC32M_en()
{
	OSC.CTRL |= OSC_RC32MEN_bm; //Wcz generator RC 32 MHz
	return OSC_wait_for_rdy(OSC_RC32MEN_bm); //Zaczekaj na jego poprawny start
}

int main(void)
{
	RC32M_en();
	CPU_CCP=CCP_IOREG_gc;            //Odblokuj zmian konfiguracji
	CLK.CTRL=CLK_SCLKSEL_RC32M_gc;   //Wybierz generator RC 32MHz

	Bufor_init();
	TimerEv_init();
	SPI_init();
	DMA_init();
		
	PMIC_CTRL=PMIC_LOLVLEN_bm; //Odblokuj przerwania niskiego poziomu
	sei();
	
	uint16_t color=0xffff;
	uint16_t rmask=0b1010101010101000;  //Maska koloru czerwonego
	for(uint8_t z=0; z<M_COLOR; z++)
	{
		color=color & rmask;
		rmask=(rmask << 2);
		for(uint8_t h=0; h<M_HEIGHT; h++)
		{
			Bufor[z][h][2]=color & 0xff;
			Bufor[z][h][3]=(color >> 8) & 0xff;
		}	
	}
	
	uint16_t gmask=0b0101010101010101;  //Maska koloru zielonego
	for(uint8_t z=1; z<M_COLOR; z++)
	{
		for(uint8_t h=0; h<(M_HEIGHT-z); h++)
		{
			Bufor[z][h][0]=gmask & 0xff;               //Cieniowanie zielonego
		    Bufor[z][h][1]=(gmask >> 8) & 0xff;
			Bufor[z][h][4]=(gmask & 0xff);             //Macierz kolorw
		    Bufor[z][h][5]=((gmask >> 8) & 0xff);
		}	
	}

	rmask=0b1111111111111101;  //Maska koloru czerwonego
	for(uint8_t z=0; z<M_COLOR; z++)
	{
		for(uint8_t h=0; h<M_HEIGHT; h++)
		{
			Bufor[z][h][4]=(Bufor[z][h][4] | 0b10101010) & (rmask & 0xff);
			Bufor[z][h][5]=(Bufor[z][h][5] | 0b10101010) & ((rmask >> 8) & 0xff);
		}	
		rmask=(rmask << 2) | 0b01;  //Nie zmieniaj stanu kanau zielonego
	}
	while(1);
}
	