/*
 * OdbiornikIR_Manchester.c
 *
 * Created: 2013-02-13 15:02:01
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>

#include "usart.h"
#include "RingBuffer.h"

#define IR_PREAMBLE_SPC 7000     //Czas od pocztku preambuy do pierwszego bitu
#define IR_HALF_BIT      889     //Czas trwania pbitu
#define IR_BIT_TOLERANCE 100     //Tolerancja czasu trwania pbitu
#define IR_BITSNO         10     //Liczba bitw kodu - 9 + 1 prebit

enum IR_NEC_States {IR_MOT_Nothing, IR_MOT_Receiving};
enum IR_NEC_States IR_State;     //Stan maszyny stanw

volatile uint16_t IR_RecCmd;     //Odebrana komenda z pilota
volatile uint8_t  IR_recpos;     //Nr aktualnie odbieranego bitu

CircBuffer IR_CMD_Buffer; //Instancja bufora koowego przechowujcego polecenia

static inline uint16_t IR_CalcTime(uint16_t time)
{
	return time*(F_CPU/1000000UL)/4;   //Przelicz czas w mikrosekundach na tyknicia timera
	                                   //Przy timerze taktowanym CLKPER/4
}

ISR(PORTD_INT0_vect)
{
	TCD0_CNT=0;
	TCD0.INTFLAGS=TC0_CCAIF_bm;        //Skasuj ewentualn flag przerwania CCA
	TCD0.INTCTRLB=TC_CCAINTLVL_LO_gc;  //Wczamy przerwania CCA
}

ISR(TCD0_OVF_vect)
{
	PORTD_INT0MASK=PORTD_INT0MASK & (~PIN2_bm); //Wycz przerwania pinu
	PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_BOTHEDGES_gc; //Wcz zdarzenia na obu zboczach
	TCD0.CTRLD=TC_EVSEL_CH1_gc | TC_EVACT_PW_gc;                //Kana 1 jako pomiar szerokoci impulsu
	TCD0.PER=IR_CalcTime(2*IR_HALF_BIT); 
	TCD0.INTFLAGS=TC0_CCAIF_bm;
	TCD0.INTCTRLB=TC_CCAINTLVL_LO_gc;  //Wczamy przerwania CCA
	IR_State=IR_MOT_Nothing;
}

ISR(TCD0_CCA_vect)
{
    static uint8_t IR_Polarity;           //Stan dekodera Manchester
	
	switch(IR_State)
	{
		case   IR_MOT_Nothing:
		        TCD0.CTRLD=TC_EVSEL_OFF_gc | TC_EVACT_OFF_gc;  //Wyczamy pomiar szerokoci impulsu
				TCD0.INTCTRLB=TC_CCAINTLVL_OFF_gc;       //Blokujemy przerwania CCA do czasu pierwszego impulsu
				uint16_t halfbittime=TCD0.CCA;           //Zapisz czas trwania pbitu
				TCD0.CCA=halfbittime+halfbittime/2;      //Przerwanie CCA ustawiomy na 1,5 czasu trwania pbitu
				TCD0.PER=IR_CalcTime(IR_PREAMBLE_SPC);   //Przeterminowanie nastpi po przekroczeniu czasu trwania preambuy
				IR_State=IR_MOT_Receiving;
				IR_recpos=0;
				IR_RecCmd=0;
				IR_Polarity=1;
				PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_FALLING_gc; //Przerwania generuje zbocze narastajce
				PORTD_INT0MASK=PIN2_bm; //Wcz przerwania pinu
		        break;
				
		case   IR_MOT_Receiving:
		        {
					IR_RecCmd|=IR_Polarity;
					uint8_t tmp=PORTD_IN;
					if(tmp & PIN2_bm)
					{
						IR_Polarity=1;
						PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_FALLING_gc; //Przerwania generuje zbocze opadajce
					} else
					{
						IR_Polarity=0;
						PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_RISING_gc; //Przerwania generuje zbocze narastajce
					}
					++IR_recpos;
					if(IR_recpos==IR_BITSNO)
					{
						PORTD_INT0MASK=~PIN2_bm; //Wycz przerwania pinu
						PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_BOTHEDGES_gc; //Wcz zdarzenia na obu zboczach
						TCD0.CTRLD=TC_EVSEL_CH1_gc | TC_EVACT_PW_gc;                //Kana 1 jako pomiar szerokoci impulsu
						IR_State=IR_MOT_Nothing;
						cb_Add(&IR_CMD_Buffer, IR_RecCmd);     //Dodaj odczytane polecenie do kolejki
					}
					IR_RecCmd<<=1;
					TCD0.PER=IR_CalcTime(2*IR_HALF_BIT); 
				}
		        break;
	}
}

void IR_init()
{
	PORTD_INTCTRL=PORT_INT0LVL_LO_gc;       //Potencjalnie wcz przerwania (pki co rejestr mask jest rwny 0)
	PORTD_PIN2CTRL=PORT_OPC_PULLUP_gc | PORT_ISC_BOTHEDGES_gc | PORT_INVEN_bm;  //Pulup na pinie odbiornika IR, oba zbocza wywouj zdarzenie
	EVSYS_CH1MUX=EVSYS_CHMUX_PORTD_PIN2_gc;                     //Transmitowane przez kana 1
	TCD0.CTRLB=TC_WGMODE_NORMAL_gc | TC0_CCAEN_bm;              //Tryb pracy normalny
	TCD0.PER=IR_CalcTime(2*IR_HALF_BIT);                        //Okres timera
	TCD0.CTRLD=TC_EVSEL_CH1_gc | TC_EVACT_PW_gc;                //Kana 1 jako pomiar szerokoci impulsu
	TCD0.INTCTRLA=TC_OVFINTLVL_LO_gc;      //Wcz przerwanie nadmiaru
	TCD0.INTCTRLB=TC_CCAINTLVL_LO_gc;      //Wcz przerwanie CCA
	TCD0.CTRLA=TC_CLKSEL_DIV4_gc;
	PMIC_CTRL=PMIC_LOLVLEN_bm;             //Odblokuj przerwania niskiego poziomu
}

void usart_init()
{
	USARTC0.CTRLB=USART_TXEN_bm;         //Wcz nadajnik USART
	USARTC0.CTRLC=USART_CHSIZE_8BIT_gc;  //Ramka 8 bitw, bez parzystoci, 1 bit stopu
	usart_set_baudrate(&USARTC0, 9600, F_CPU);
	PORTC_DIRSET=PIN3_bm;                //Pin TxD musi by wyjciem
}

int main(void)
{
	char bufor[13];
	
	IR_init();
	usart_init();
	sei();
	
	while(1)
	{
		if(cb_IsEmpty(&IR_CMD_Buffer)==false)
		{
			CB_Element cmd=cb_Read(&IR_CMD_Buffer);
			ultoa(cmd, bufor, 16);
			USART_send(&USARTC0, bufor);
			USART_putchar(&USARTC0, '\n');
			USART_putchar(&USARTC0, '\r');
		}
	}
}
