// LISTING 10.8. Klasa MyPostfunction z metodą wykonującą kod po wykonaniu przejścia w workflow

package pl.helion.jira.workflow;

import java.util.Map;

import com.atlassian.jira.bc.issue.IssueService;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.config.SubTaskManager;
import com.atlassian.jira.exception.CreateException;
import com.atlassian.jira.issue.IssueInputParameters;
import com.atlassian.jira.issue.MutableIssue;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.workflow.function.issue.AbstractJiraFunctionProvider;
import com.opensymphony.module.propertyset.PropertySet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Klasa zawiera metodę execute(), w której ciele umieszcza się kod do wykonania po wykonaniu przejścia w workflow.
 * W mapie args znajdują się zmienne załadowane w metodzie getDescriptorParams() z klasy MyPostFunctionFactory.
 */
public class MyPostfunction extends AbstractJiraFunctionProvider {

    // 1. Pobranie loggera.
    private static final Logger log = LoggerFactory.getLogger(MyPostfunction.class);

    /**
     * 2. Metoda wykonująca skrypt post funkcji po wykonaniu przejścia.
     * @param transientVars — tablica przechowująca dane przejścia.
     * @param args          — parametry załadowane w MyPostfunctionFactory.
     * @param ps            — property nie jest wykorzystywane.
     */
    public void execute(Map transientVars, Map args, PropertySet ps) {
        
        // 3. Pobranie obiektu typu interfejsowego Issue z danych przejścia i zrzutowanie go do typu interfejsowego MutableIssue.
        MutableIssue issue = getIssue(transientVars);
        
        // 4. Pobranie nazwy użytkownika Assignee ze zgłoszenia, na którym jest wykonywane przejście.
        String assigneeUserName = issue.getAssignee() == null ? "" : issue.getAssignee().getName();

        // 5. Pobranie obiektu typu interfejsowego ApplicationUser aktualnie zalogowanego użytkownika.
        ApplicationUser currentUser = ComponentAccessor.getJiraAuthenticationContext().getLoggedInUser();

        // 6. Pobranie wartości Summary dla nowo tworzonego sub-tasku.
        String subtaskSummary = (String) args.get("subtaskSummary");

        // 7. Pobranie komponentu issueService.
        IssueService issueService = ComponentAccessor.getIssueService();

        // 8. Inicjalizacja listy parametrów dla nowego sub-tasku.
        IssueInputParameters inputParameters = issueService.newIssueInputParameters();

        // 9. Uzupełnienie parametrów dla sub-tasku.
        inputParameters
                .setProjectId(issue.getProjectId())
                .setSummary(subtaskSummary)
                .setIssueTypeId("10003")
                .setAssigneeId(assigneeUserName)
                .setApplyDefaultValuesWhenParameterNotProvided(true);

        // 10. Walidacja parametrów sub-tasku.
        IssueService.CreateValidationResult subTaskValidationResult = issueService.validateSubTaskCreate(currentUser, issue.getId(), inputParameters);

        // 11. Zapisanie logu ewentualnego błędu walidacji parametrów.
        if (!subTaskValidationResult.isValid())
            subTaskValidationResult.getErrorCollection().getErrors().forEach(
                    (key, value) -> {
                        log.warn(value.toString());
                    });

        // 12. Pobranie komponentu SubTaskManager i utworzenie sub-tasku.
        SubTaskManager subTaskManager = ComponentAccessor.getSubTaskManager();
        IssueService.IssueResult subTask = issueService.create(currentUser, subTaskValidationResult);
        
        // 13. Utworzenie linku między zgłoszeniem nadrzędnym i nowo utworzonym sub-taskiem.
        try {
            subTaskManager.createSubTaskIssueLink(issue, subTask.getIssue(), issue.getReporterUser());
        } catch (CreateException e) {
            throw new RuntimeException(e);
        }
    }
}