﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Rendering;
using Microsoft.EntityFrameworkCore;
using TriathlonRaceTracking.Data;
using TriathlonRaceTracking.Models;

namespace TriathlonRaceTracking.Controllers
{
    public class RegistrationsController : Controller
    {
        private readonly TriathlonRaceTrackingContext _context;

        public RegistrationsController(TriathlonRaceTrackingContext context)
        {
            _context = context;
        }

        // GET: Registrations
        public async Task<IActionResult> Index()
        {
            var triathlonRaceTrackingContext = _context.Registrations.Include(r => r.Athlete).Include(r => r.Race);
            return View(await triathlonRaceTrackingContext.ToListAsync());
        }

        // GET: Registrations/Details/5
        public async Task<IActionResult> Details(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var registration = await _context.Registrations
                .Include(r => r.Athlete)
                .Include(r => r.Race)
                .SingleOrDefaultAsync(m => m.ID == id);
            if (registration == null)
            {
                return NotFound();
            }

            return View(registration);
        }

        // GET: Registrations/Create
        public IActionResult Create()
        {
            ViewData["AthleteID"] = new SelectList(_context.Athletes, "ID", "FullName");
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name");
            return View();
        }

        // POST: Registrations/Create
        // Aby chronić przed atakami należy włączyć konkretne właściwości, z którymi ma następować łączenie.
        // Więcej informacji na ten temat znajdziesz na stronie http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Create([Bind("ID,AthleteID,RaceID,BibNumber")] Registration registration)
        {
            if (ModelState.IsValid)
            {
                _context.Add(registration);
                await _context.SaveChangesAsync();
                return RedirectToAction(nameof(Index));
            }
            ViewData["AthleteID"] = new SelectList(_context.Athletes, "ID", "FullName", registration.AthleteID);
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name", registration.RaceID);
            return View(registration);
        }

        // GET: Registrations/Edit/5
        public async Task<IActionResult> Edit(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var registration = await _context.Registrations.SingleOrDefaultAsync(m => m.ID == id);
            if (registration == null)
            {
                return NotFound();
            }
            ViewData["AthleteID"] = new SelectList(_context.Athletes, "ID", "FullName", registration.AthleteID);
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name", registration.RaceID);
            return View(registration);
        }

        // POST: Registrations/Edit/5
        // Aby chronić przed atakami należy włączyć konkretne właściwości, z którymi ma następować łączenie. 
        // Więcej informacji na ten temat znajdziesz na stronie http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Edit(int id, [Bind("ID,AthleteID,RaceID,BibNumber")] Registration registration)
        {
            if (id != registration.ID)
            {
                return NotFound();
            }

            if (ModelState.IsValid)
            {
                try
                {
                    _context.Update(registration);
                    await _context.SaveChangesAsync();
                }
                catch (DbUpdateConcurrencyException)
                {
                    if (!RegistrationExists(registration.ID))
                    {
                        return NotFound();
                    }
                    else
                    {
                        throw;
                    }
                }
                return RedirectToAction(nameof(Index));
            }
            ViewData["AthleteID"] = new SelectList(_context.Athletes, "ID", "FullName", registration.AthleteID);
            ViewData["RaceID"] = new SelectList(_context.Races, "ID", "Name", registration.RaceID);
            return View(registration);
        }

        // GET: Registrations/Delete/5
        public async Task<IActionResult> Delete(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var registration = await _context.Registrations
                .Include(r => r.Athlete)
                .Include(r => r.Race)
                .SingleOrDefaultAsync(m => m.ID == id);
            if (registration == null)
            {
                return NotFound();
            }

            return View(registration);
        }

        // POST: Registrations/Delete/5
        [HttpPost, ActionName("Delete")]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> DeleteConfirmed(int id)
        {
            var registration = await _context.Registrations.SingleOrDefaultAsync(m => m.ID == id);
            _context.Registrations.Remove(registration);
            await _context.SaveChangesAsync();
            return RedirectToAction(nameof(Index));
        }

        private bool RegistrationExists(int id)
        {
            return _context.Registrations.Any(e => e.ID == id);
        }
    }
}
