﻿using Microsoft.VisualStudio.TestTools.UnitTesting;
using Moq;
using SportsStore.Domain.Abstract;
using SportsStore.Domain.Entities;
using SportsStore.WebUI.Controllers;
using SportsStore.WebUI.Models;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using SportsStore.WebUI.HtmlHelpers;

namespace SportsStore.UnitTests {
    [TestClass]
    public class UnitTest1 {

[TestMethod]
public void Can_Paginate() {

    // przygotowanie
    Mock<IProductRepository> mock = new Mock<IProductRepository>();
    mock.Setup(m => m.Products).Returns(new Product[] {
        new Product {ProductID = 1, Name = "P1"},
        new Product {ProductID = 2, Name = "P2"},
        new Product {ProductID = 3, Name = "P3"},
        new Product {ProductID = 4, Name = "P4"},
        new Product {ProductID = 5, Name = "P5"}
    }.AsQueryable());

    // utworzenie kontrolera i ustawienie 3-elementowej strony
    ProductController controller = new ProductController(mock.Object);
    controller.PageSize = 3;

    // działanie
    ProductsListViewModel result = (ProductsListViewModel)controller.List(null, 2).Model;

    // asercje
    Product[] prodArray = result.Products.ToArray();
    Assert.IsTrue(prodArray.Length == 2);
    Assert.AreEqual(prodArray[0].Name, "P4");
    Assert.AreEqual(prodArray[1].Name, "P5");
}

        [TestMethod]
        public void Can_Generate_Page_Links() {

            // przygotowanie — definiowanie metody pomocniczej HTML — potrzebujemy tego,
            // aby użyć metody rozszerzającej
            HtmlHelper myHelper = null;

            // przygotowanie — tworzenie danych PagingInfo
            PagingInfo pagingInfo = new PagingInfo {
                CurrentPage = 2,
                TotalItems = 28,
                ItemsPerPage = 10
            };

            // przygotowanie — konfigurowanie delegatu z użyciem wyrażenia lambda
            Func<int, string> pageUrlDelegate = i => "Strona" + i;

            // działanie
            MvcHtmlString result = myHelper.PageLinks(pagingInfo, pageUrlDelegate);

            // asercje
            Assert.AreEqual(result.ToString(), @"<a href=""Strona1"">1</a>" 
                + @"<a class=""selected"" href=""Strona2"">2</a>" 
                + @"<a href=""Strona3"">3</a>");
        }

        [TestMethod]
        public void Can_Send_Pagination_View_Model() {

            // przygotowanie
            Mock<IProductRepository> mock = new Mock<IProductRepository>();
            mock.Setup(m => m.Products).Returns(new Product[] {
                new Product {ProductID = 1, Name = "P1"},
                new Product {ProductID = 2, Name = "P2"},
                new Product {ProductID = 3, Name = "P3"},
                new Product {ProductID = 4, Name = "P4"},
                new Product {ProductID = 5, Name = "P5"}
            }.AsQueryable());

            // przygotowanie - utworzenie kontrolera i ustawienie 3-elementowej strony
            ProductController controller = new ProductController(mock.Object);
            controller.PageSize = 3;

            // działanie
            ProductsListViewModel result = (ProductsListViewModel)controller.List(null, 2).Model;

            // asercje
            PagingInfo pageInfo = result.PagingInfo;
            Assert.AreEqual(pageInfo.CurrentPage, 2);
            Assert.AreEqual(pageInfo.ItemsPerPage, 3);
            Assert.AreEqual(pageInfo.TotalItems, 5);
            Assert.AreEqual(pageInfo.TotalPages, 2);
        }

        [TestMethod]
        public void Can_Filter_Products() {

            // przygotowanie
            // - utworzenie imitacji repozytorium
            Mock<IProductRepository> mock = new Mock<IProductRepository>();
            mock.Setup(m => m.Products).Returns(new Product[] {
                new Product {ProductID = 1, Name = "P1", Category = "Cat1"},
                new Product {ProductID = 2, Name = "P2", Category = "Cat2"},
                new Product {ProductID = 3, Name = "P3", Category = "Cat1"},
                new Product {ProductID = 4, Name = "P4", Category = "Cat2"},
                new Product {ProductID = 5, Name = "P5", Category = "Cat3"}
            }.AsQueryable());

            // przygotowanie - utworzenie kontrolera i ustawienie 3-elementowej strony
            ProductController controller = new ProductController(mock.Object);
            controller.PageSize = 3;

            // działanie
            Product[] result = ((ProductsListViewModel)controller.List("Cat2", 1).Model)
                .Products.ToArray();

            // asercje
            Assert.AreEqual(result.Length, 2);
            Assert.IsTrue(result[0].Name == "P2" && result[0].Category == "Cat2");
            Assert.IsTrue(result[1].Name == "P4" && result[1].Category == "Cat2");
        }

        [TestMethod]
        public void Can_Create_Categories() {

            // przygotowanie
            // - utworzenie imitacji repozytorium
            Mock<IProductRepository> mock = new Mock<IProductRepository>();
            mock.Setup(m => m.Products).Returns(new Product[] {
                new Product {ProductID = 1, Name = "P1", Category = "jabłka"},
                new Product {ProductID = 2, Name = "P2", Category = "jabłka"},
                new Product {ProductID = 3, Name = "P3", Category = "gruszki"},
                new Product {ProductID = 4, Name = "P4", Category = "pomarańcze"},
            }.AsQueryable());

            // przygotowanie - utworzenie kontrolera
            NavController target = new NavController(mock.Object);

            // działanie - pobranie zbioru kategorii 
            string[] results = ((IEnumerable<string>)target.Menu().Model).ToArray();

            // asercje
            Assert.AreEqual(results.Length, 3);
            Assert.AreEqual(results[0], "jabłka");
            Assert.AreEqual(results[1], "pomarańcze");
            Assert.AreEqual(results[2], "gruszki");
        }

        [TestMethod]
        public void Indicates_Selected_Category() {

            // przygotowanie
            // - utworzenie imitacji repozytorium
            Mock<IProductRepository> mock = new Mock<IProductRepository>();
            mock.Setup(m => m.Products).Returns(new Product[] {
                new Product {ProductID = 1, Name = "P1", Category = "jabłka"},
                new Product {ProductID = 4, Name = "P2", Category = "pomarańcze"},
            }.AsQueryable());

            // przygotowanie - utworzenie kontrolera 
            NavController target = new NavController(mock.Object);

            // przygotowanie - definiowanie kategorii do wybrania
            string categoryToSelect = "jabłka";

            // działanie
            string result = target.Menu(categoryToSelect).ViewBag.SelectedCategory;

            // asercje
            Assert.AreEqual(categoryToSelect, result);
        }

        [TestMethod]
        public void Generate_Category_Specific_Product_Count() {
            // przygotowanie
            // - utworzenie imitacji repozytorium
            Mock<IProductRepository> mock = new Mock<IProductRepository>();
            mock.Setup(m => m.Products).Returns(new Product[] {
                new Product {ProductID = 1, Name = "P1", Category = "Cat1"},
                new Product {ProductID = 2, Name = "P2", Category = "Cat2"},
                new Product {ProductID = 3, Name = "P3", Category = "Cat1"},
                new Product {ProductID = 4, Name = "P4", Category = "Cat2"},
                new Product {ProductID = 5, Name = "P5", Category = "Cat3"}
            }.AsQueryable());

            // przygotowanie - utworzenie kontrolera i ustawienie 3-elementowej strony
            ProductController target = new ProductController(mock.Object);
            target.PageSize = 3;

            // działanie - testowanie liczby produktów dla różnych kategorii
            int res1 
                = ((ProductsListViewModel)target.List("Cat1").Model).PagingInfo.TotalItems;
            int res2 
                = ((ProductsListViewModel)target.List("Cat2").Model).PagingInfo.TotalItems;
            int res3 
                = ((ProductsListViewModel)target.List("Cat3").Model).PagingInfo.TotalItems;
            int resAll 
                = ((ProductsListViewModel)target.List(null).Model).PagingInfo.TotalItems;

            // asercje
            Assert.AreEqual(res1, 2);
            Assert.AreEqual(res2, 2);
            Assert.AreEqual(res3, 1);
            Assert.AreEqual(resAll, 5);
        }


    }
}
