﻿using PortalUslug.Models;
using System.Linq;
using System.Web.Security;
using WebMatrix.WebData;

namespace PortalUslug.Repositories
{
    /// <summary>
    ///  Klasa repozytorium usługodawcy.
    /// </summary>
    public class ServiceProviderRepository : IServiceProviderRepository
    {
        /// <summary>
        /// Obiekt klasy kontekstowej.
        /// </summary>
        private PortalUslugContext _db;

        /// <summary>
        /// Konstruktor repozytorium usługodawcy.
        /// </summary>
        public ServiceProviderRepository()
        {
            _db = new PortalUslugContext();
        }

        /// <summary>
        /// Pobranie usługodawcy o podanym identyfikatorze usługodawcy.
        /// </summary>
        /// <param name="id">Identyfikator usługodawcy.</param>
        /// <returns>Usługodawca o podanym identyfikatorze usługodawcy.</returns>
        public ServiceProvider GetServiceProviderById(int id)
        {
            return _db.ServiceProviders.FirstOrDefault(provider => provider.Id == id);
        }

        /// <summary>
        /// Pobranie usługodawcy o podanym identyfikatorze użytkownika.
        /// </summary>
        /// <param name="id">Identyfikator użytkownika.</param>
        /// <returns>Usługodawca o podanym identyfikatorze użytkownika.</returns>
        public ServiceProvider GetServiceProviderByUserId(int id)
        {
            return _db.ServiceProviders.FirstOrDefault(provider => provider.UserId == id);
        }

        /// <summary>
        /// Pobieranie usługodawców z potwierdzonym kontem.
        /// </summary>
        /// <returns>Usługodawcy z potwierdzonym kontem.</returns>
        public IQueryable<ServiceProvider> GetServiceProviders()
        {
            return _db.ServiceProviders.Where(u => u.IsConfirmed);
        }

        /// <summary>
        /// Pobieranie usługodawców z newsletterem.
        /// </summary>
        /// <returns>Usługodawcy z newsletterem.</returns>
        public IQueryable<ServiceProvider> GetAllServiceProvidersWithNewsletter()
        {
            return _db.ServiceProviders.Where(u => u.Newsletter && u.IsConfirmed);
        }

        /// <summary>
        /// Sprawdzenie, czy użytkownik o podanym identyfikatorze jest usługodawcą.
        /// </summary>
        /// <param name="id">Identyfikator użytkownika.</param>
        /// <returns>True, jeśli użytkownik o podanym identyfikatorze jest usługodawcą.</returns>
        public bool IsServiceProvider(int id)
        {
            return _db.ServiceProviders.FirstOrDefault(u => u.UserId == id) != null ? true : false;
        }

        /// <summary>
        /// Sprawdzenie, czy użytkownik o podanym identyfikatorze ma potwierdzone konto.
        /// </summary>
        /// <param name="id">Identyfikator użytkownika.</param>
        /// <returns>True, jeśli użytkownik o podanym identyfikatorze ma potwierdzone konto.</returns>
        public bool IsConfirmed(int id)
        {
            return _db.ServiceProviders.FirstOrDefault(u => u.UserId == id).IsConfirmed;
        }

        /// <summary>
        /// Dodanie usługodawcy.
        /// </summary>
        /// <param name="provider">Dodawany usługodawca.</param>
        public void Add(ServiceProvider provider)
        {
            _db.ServiceProviders.Add(provider);
        }

        /// <summary>
        /// Usunięcie usługodawcy.
        /// </summary>
        /// <param name="provider">Usuwany usługodawca.</param>
        public void Delete(ServiceProvider provider)
        {
            string name = ((SimpleMembershipProvider)Membership.Provider).GetUserNameFromId(provider.UserId);
            _db.ServiceProviders.Remove(provider);
            _db.SaveChanges();
                        
            Roles.RemoveUserFromRole(name, "ServiceProvider");
            ((SimpleMembershipProvider)Membership.Provider).DeleteAccount(name);
            ((SimpleMembershipProvider)Membership.Provider).DeleteUser(name, true);        
        }

        /// <summary>
        /// Zapisanie zmian.
        /// </summary>
        public void SaveChanges()
        {
            _db.SaveChanges();
        }

        /// <summary>
        /// Pobranie wszystkich usługodawców.
        /// </summary>
        /// <returns>Usługodawcy.</returns>
        public IQueryable<ServiceProviderViewModel> GetAllServiceProviders()
        {
            var providers = from p in _db.ServiceProviders
                            select new ServiceProviderViewModel
                            {
                                Id = p.Id,
                                Name = p.Name,
                                City = p.City,
                                Street = p.Street,
                                ZipCode = p.ZipCode,
                                UserId = p.UserId,
                                PhoneNumber = p.PhoneNumber,
                                IsActive = p.IsConfirmed ? "Tak" : "Nie",
                                IsConfirmed = p.IsConfirmed,
                                RegistrationDate = p.RegistrationDate
                            };

            return providers;
        }
    }
}