﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Configuration;
using System.Linq;
using System.Web.Mvc;
using System.Web.Security;
using MvcContrib.Pagination;
using MvcContrib.Sorting;
using MvcContrib.UI.Grid;
using PortalUslug.Models;
using PortalUslug.Repositories;
using WebMatrix.WebData;

namespace PortalUslug.Controllers
{
    public class ServiceController : Controller
    {
        /// <summary>
        /// Zmienna przechowująca parametr odczytany z pliku konfiguracyjnego.
        /// Domyślna liczba dni, przez jaką usługa będzie dostępna w serwisie.
        /// </summary>
        int daysNumber = 0;

        /// <summary>
        /// Repozytorium usług.
        /// </summary>
        private ServiceRepository _serviceRepo;

        /// <summary>
        /// Repozytorium kategorii.
        /// </summary>
        private CategoryRepository _categoryRepo;

        /// <summary>
        /// Repozytorium usługodawców.
        /// </summary>
        private ServiceProviderRepository _providerRepo;

        /// <summary>
        /// Repozytorium usługobiorców.
        /// </summary>
        private CustomerRepository _customerRepo;

        /// <summary>
        /// Repozytorium komentarzy.
        /// </summary>
        private CommentRepository _commentRepo;

        /// <summary>
        /// Konstruktor kontrolera usług.
        /// </summary>
        public ServiceController()
        {
            // Inicjalizacja repozytoriów 
            _serviceRepo = new ServiceRepository();
            _categoryRepo = new CategoryRepository();
            _providerRepo = new ServiceProviderRepository();
            _customerRepo = new CustomerRepository();
            _commentRepo = new CommentRepository();

            // Odczytanie domyślnego parametru liczby dni aktywności usługi z pliku konfiguracyjnego
            if (!int.TryParse(ConfigurationManager.AppSettings["DaysNumber"], out daysNumber))
            {
                // Jeżeli wartość paramteru nie jest zdefiniowana w pliku konfiguracyjnym, to wartość ustawiana jest na 7.
                daysNumber = 7;
            }
        }

        //
        // GET: /Service/

        public ActionResult Index(string name, string content, int? serviceProviderId, int? categoryId, GridSortOptions gridSortOptions, [DefaultValue(1)]int page)
        {
            // Pobranie listy aktywnych usług
            var servicesList = _serviceRepo.GetActiveService();

            // Jeżeli zalogowany użytkownik jest administratorem, to należy pobrać listę wszystkich usług.
            if (!Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("Administrator"))
            {
                servicesList = _serviceRepo.GetAllService();
            }

            // Ustawienie domyślnej kolumny sortowania
            if (string.IsNullOrWhiteSpace(gridSortOptions.Column))
            {
                gridSortOptions.Column = "Id";
            }

            // Filtrowanie po usługodawcy
            if (serviceProviderId.HasValue)
            {
                servicesList = servicesList.Where(a => a.UserId == serviceProviderId);
            }

            // Filtrowanie po kategorii
            if (categoryId.HasValue)
            {
                servicesList = servicesList.Where(a => a.CategoryId == categoryId);
            }

            // Filtrowanie po nazwie
            if (!string.IsNullOrWhiteSpace(name))
            {
                servicesList = servicesList.Where(a => a.Name.Contains(name));
            }

            // Filtrowanie po treści
            if (!string.IsNullOrWhiteSpace(content))
            {
                servicesList = servicesList.Where(a => a.Content.Contains(content));
            }

            // Przygotowanie modelu do filtrowania, wypełnienie modelu danymi
            var serviceFilterViewModel = new ServiceFilterViewModel();
            serviceFilterViewModel.SelectedCategoryId = categoryId ?? -1;
            serviceFilterViewModel.SelectedProviderId = serviceProviderId ?? -1;
            serviceFilterViewModel.Fill();

            // Sortowanie listy usług oraz stronicowanie
            var servicePagedList = servicesList.OrderBy(gridSortOptions.Column, gridSortOptions.Direction).AsPagination(page, 10);

            var serviceListContainer = new ServiceListContainerViewModel
            {
                ServicePagedList = servicePagedList,
                ServiceFilterViewModel = serviceFilterViewModel,
                GridSortOptions = gridSortOptions
            };

            return View(serviceListContainer);
        }

        //
        // GET: /Service/Details/5

        public ActionResult Details(int id, [DefaultValue(1)]int page)
        {
            bool isConfirmed = false;

            // Pobranie usługi po identyfikatorze
            var service = _serviceRepo.GetServiceViewModelById(id);

            // Pobranie listy komentarzy danej usługi
            var comments = _commentRepo.GetCommentByServiceId(id);

            // Jeżeli użytkownik jest zalogowany i ma potwierdzone konto, to będzie widzieć listę komentarzy.
                var userId = WebSecurity.CurrentUserId;

                // Sprawdzenie, czy użytkownik potwierdził konto
                if (_providerRepo.IsServiceProvider(userId))
                {
                    isConfirmed = _providerRepo.IsConfirmed(userId);
                }
                else if (_customerRepo.IsCustomer(userId))
                {
                    isConfirmed = _customerRepo.IsConfirmed(userId);
                }

                // Sortowanie listy komentarzy oraz stronicowanie
            var commentsList = comments.OrderBy("Date", MvcContrib.Sorting.SortDirection.Ascending).AsPagination(page, 5);

            var serviceComments = new ServiceCommentsViewModel
            {
                Service = service,
                CommentPagedList = commentsList,
                ConfirmedUser = isConfirmed,
            };

            return View(serviceComments);
        }

        //
        // GET: /Service/Create

        [Authorize(Roles = "ServiceProvider")]
        public ActionResult Create()
        {
            // Pobranie listy kategorii usług i umieszczenie jej w obiekcie ViewData
            ViewData["Categories"] = GetCategoriesList();
            return View();
        }

        //
        // POST: /Service/Create

        [Authorize(Roles = "ServiceProvider")]
        [HttpPost]
        public ActionResult Create(AddServiceModel service)
        {
            if (ModelState.IsValid)
            {
                // Dodanie nowej usługi i zapisanie w bazie danych
                if (TryUpdateModel(service))
                {
                    // Ustawienie daty umieszczenia usługi
                    service.Service.PostedDate = DateTime.Now;

                    // Ustawienie daty wygaśnięcia usługi
                    if (service.DaysNumber.HasValue)
                    {
                        service.Service.ExpirationDate = service.Service.PostedDate.AddDays(service.DaysNumber.Value);
                    }
                    else
                    {
                        service.Service.ExpirationDate = service.Service.PostedDate.AddDays(daysNumber);
                    }

                    // Ustawienie użytkownika dodającego usługę oraz adresu IP
                    service.Service.UserId = WebSecurity.CurrentUserId;
                    service.Service.IPAddress = Request.UserHostAddress;

                    // Dodanie nowej usługi i zapisanie zmian
                    _serviceRepo.Add(service.Service);
                    _serviceRepo.SaveChanges();

                    return RedirectToAction("Details", new { id = service.Service.Id });
                }

                // Pobranie listy kategorii usług i umieszczenie jej w obiekcie ViewData  
                ViewData["Categories"] = GetCategoriesList();
                return View(service);
            }

            TempData["Error"] = "Wystąpił błąd podczas dodawania usługi!";
            return RedirectToAction("Index");
        }

        [NonAction]
        private List<SelectListItem> GetCategoriesList()
        {
            // Pobranie listy kategorii usług
            var categoriesList = _categoryRepo.GetAllCategories();
            var categories = new List<SelectListItem>();

            // Wygenerowanie listy w postaci: nazwa – ID kategorii
            foreach (var category in categoriesList)
            {
                categories.Add(new SelectListItem() { Text = category.Name, Value = category.Id.ToString() });
            }

            return categories;
        }

        //
        // GET: /Service/Edit/5

        [Authorize(Roles = "ServiceProvider, Administrator")]
        public ActionResult Edit(int id = 0)
        {
            // Pobranie usługi po identyfikatorze
            var service = _serviceRepo.GetServiceById(id);

            if (service != null)
            {
                // Pobranie listy kategorii usług i umieszczenie jej w obiekcie ViewData
                ViewData["Categories"] = GetEditCategories(service);

                // Utworzenie modelu widokowego w celu przekazania go do widoku
                AddServiceModel editService = new AddServiceModel();
                editService.Service = service;
                                
                // Sprawdzenie, czy zalogowany użytkownik może edytować usługę
                    if (Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("ServiceProvider") || Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("Administrator"))
                {
                    if (service.UserId == WebSecurity.CurrentUserId)
                    {
                        return View(editService);
                    }
                    else if (Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("Administrator"))
                    {
                        return View(editService);
                    }
                    else
                    {
                        TempData["Error"] = "Nie masz uprawnień do edytowania tej usługi.";
                        return RedirectToAction("Index");
                    }
                }
                else
                {
                    TempData["Error"] = "Nie masz uprawnień do edytowania tej usługi.";
                    return RedirectToAction("Index");
                }
            }
            else
            {
                TempData["Error"] = "Brak usługi o podanym ID!";
                return RedirectToAction("Index");
            }
        }

        //
         // POST: /Service/Edit/5

        [Authorize(Roles = "ServiceProvider, Administrator")]
        [HttpPost]
        public ActionResult Edit(int id, AddServiceModel editService)
        {
            // Pobranie usługi po identyfikatorze
            var service = _serviceRepo.GetServiceById(id);
            editService.Service = service;

            // Wyliczenie daty wygaśnięcia usługi na podstawie wartości z modelu lub z pliku konfiguracyjnego
            if (editService.DaysNumber.HasValue)
            {
                editService.Service.ExpirationDate = service.PostedDate.AddDays(editService.DaysNumber.Value);
            }
            else
            {
                editService.Service.ExpirationDate = service.PostedDate.AddDays(daysNumber);
            }

            // Uaktualnienie modelu i zapisanie zmian w bazie danych
            if (TryUpdateModel(editService))
            {
                _serviceRepo.SaveChanges();
                TempData["Message"] = "Pomyślnie zmodyfikowano usługę!";
                return RedirectToAction("Details", new { id = editService.Service.Id });
            }
            else
            {
                // Pobranie listy kategorii usług i umieszczenie jej w obiekcie ViewData
                ViewData["Categories"] = GetEditCategories(editService.Service);
                TempData["Error"] = "Wystąpił błąd podczas edytowania usługi!";
                return View(editService);
            }
        }

        [NonAction]
        private List<SelectListItem> GetEditCategories(Service service)
        {
            // Pobranie listy kategorii usług
            var categoriesList = _categoryRepo.GetAllCategories();
            var categories = new List<SelectListItem>();

            // Wygenerowanie listy w postaci: nazwa - ID kategorii oraz zaznaczenie wybranej w usłudze kategorii
            foreach (var category in categoriesList)
            {
                categories.Add(new SelectListItem
                {
                    Text = category.Name,
                    Value = category.Id.ToString(),
                    Selected = category.Id == service.CategoryId

                });
            }
            return categories;
        }

        //
        // GET: /Service/Delete/5

        [Authorize(Roles = "ServiceProvider, Administrator")]
        public ActionResult Delete(int id = 0)
        {
            // Pobranie usługi po identyfikatorze
            var service = _serviceRepo.GetServiceById(id);

            if (service != null)
            {
                // Sprawdzenie, czy do usługi zostały dodane komentarze
                if (service.Comments.Count == 0)
                {
                    // Usługę może usunąć jedynie administrator lub usługodawca, który ją dodał
                    if (Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("ServiceProvider") || Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("Administrator"))
                    {
                        if (service.UserId == WebSecurity.CurrentUserId)
                        {
                            return View(service);
                        }
                        else if (Roles.GetRolesForUser(WebSecurity.CurrentUserName).Contains("Administrator"))
                        {
                            return View(service);
                        }
                        else
                        {
                            TempData["Error"] = "Nie masz uprawnień do usunięcia tej usługi.";
                            return RedirectToAction("Index");
                        }
                    }
                    else
                    {
                        TempData["Error"] = "Nie masz uprawnień do usunięcia tej usługi.";
                        return RedirectToAction("Index");
                    }
                }
                else
                {
                    TempData["Error"] = "Nie można usunąć wybranej usługi.";
                    return RedirectToAction("Index");
                }
            }
            else
            {
                TempData["Error"] = "Brak usługi o podanym ID!";
                return RedirectToAction("Index");
            }
        }

        //
        // POST: /Service/Delete/5

        [Authorize(Roles = "ServiceProvider, Administrator")]
        [HttpPost]
        public ActionResult Delete(int id, FormCollection collection)
        {
            // Pobranie usługi po identyfikatorze
            var service = _serviceRepo.GetServiceById(id);

            try
            {
                // Usunięcie usługi i zapisanie zmian w bazie danych
                _serviceRepo.Delete(service);
                _serviceRepo.SaveChanges();
            }
            catch (Exception)
            {
                TempData["Error"] = "Nie można usunąć tej usługi.";
                return View(service);
            }

            TempData["Message"] = "Usługa została usunięta.";
            return RedirectToAction("Index");
        }
    }
}
