﻿using System;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Identity;
using Microsoft.Extensions.Configuration;
using TestMakerFreeWebApp.ViewModels;
using TestMakerFreeWebApp.Data;
using Mapster;

namespace TestMakerFreeWebApp.Controllers
{
   public class UserController : BaseApiController
   {
      #region Konstruktor
      public UserController(
          ApplicationDbContext context,
          RoleManager<IdentityRole> roleManager,
          UserManager<ApplicationUser> userManager,
          IConfiguration configuration
          )
          : base(context, roleManager, userManager, configuration) { }
      #endregion

      #region Metody dostosowujące do konwencji REST
      /// <summary>
      /// PUT: api/user
      /// </summary>
      /// <returns>Tworzy i zwraca nowego użytkownika</returns>
      [HttpPut()]
      public async Task<IActionResult> Put([FromBody]UserViewModel model)
      {
         // Zwraca ogólny kod statusu HTTP 500 (Server Error),
         // jeśli dane przesłane przez klienta są niewłaściwe
         if (model == null) return new StatusCodeResult(500);

         // Sprawdź, czy nazwa użytkownika lub e-mail już istnieje
         ApplicationUser user = await UserManager.FindByNameAsync(model.UserName);
         if (user != null) return BadRequest("Nazwa użytkownika jest już używana");

         user = await UserManager.FindByEmailAsync(model.Email);
         if (user != null) return BadRequest("Adres e-mail jest już używany");

         var now = DateTime.Now;

         // Utwórz nowego użytkownika na podstawie danych przesłanych przez klienta
         user = new ApplicationUser()
         {
            SecurityStamp = Guid.NewGuid().ToString(),
            UserName = model.UserName,
            Email = model.Email,
            DisplayName = model.DisplayName,
            CreatedDate = now,
            LastModifiedDate = now
         };

         // Dodaj użytkownika do bazy danych z wybranym hasłem
         await UserManager.CreateAsync(user, model.Password);

         // Przypisz użytkownikowi rolę 'ZarejestrowanyUżytkownik'.
         await UserManager.AddToRoleAsync(user, "ZarejestrowanyUżytkownik");

         // Usuń potwierdzanie e-maila i blokadę
         user.EmailConfirmed = true;
         user.LockoutEnabled = false;

         // Zapisz zmiany w bazie danych
         DbContext.SaveChanges();

         // Przekaż nowoutworzonego użytkownika klientowi
         return Json(user.Adapt<UserViewModel>(),
             JsonSettings);
      }
      #endregion
   }
}
