using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.SpaServices.Webpack;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.EntityFrameworkCore;
using TestMakerFreeWebApp.Data;
using Microsoft.AspNetCore.Identity;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.AspNetCore.Authentication.JwtBearer;
using Microsoft.IdentityModel.Tokens;
using System.Text;


namespace TestMakerFree
{
   public class Startup
   {
      public Startup(IConfiguration configuration)
      {
         Configuration = configuration;
      }

      public IConfiguration Configuration { get; }

      // This method gets called by the runtime. Use this method to add services to the container
      public void ConfigureServices(IServiceCollection services)
      {
         services.AddMvc();

         // Dodanie obsugi EntityFramework zwizanej z SqlServer
         services.AddEntityFrameworkSqlServer();

         // Dodanie ApplicationDbContext
         services.AddDbContext<ApplicationDbContext>(options =>
            options.UseSqlServer(Configuration.GetConnectionString("DefaultConnection"))
         );

         // Dodanie obsugi ASP.NET Identity
         services.AddIdentity<ApplicationUser, IdentityRole>(
         opts =>
         {
            opts.Password.RequireDigit = true;
            opts.Password.RequireLowercase = true;
            opts.Password.RequireUppercase = true;
            opts.Password.RequireNonAlphanumeric = false;
            opts.Password.RequiredLength = 7;
         })
         .AddEntityFrameworkStores<ApplicationDbContext>();

         // Dodaj uwierzytelnianie
         services.AddAuthentication(opts =>
         {
            opts.DefaultScheme = JwtBearerDefaults.AuthenticationScheme;
            opts.DefaultAuthenticateScheme = JwtBearerDefaults.AuthenticationScheme;
            opts.DefaultChallengeScheme = JwtBearerDefaults.AuthenticationScheme;
         })
         // Dodaj obsug tokena JWT
         .AddJwtBearer(cfg =>
         {
            cfg.RequireHttpsMetadata = false;
            cfg.SaveToken = true;
            cfg.TokenValidationParameters = new TokenValidationParameters()
            {
               // Konfiguracja standardowa
               ValidIssuer = Configuration["Auth:Jwt:Issuer"],
               IssuerSigningKey = new SymmetricSecurityKey(
                       Encoding.UTF8.GetBytes(Configuration["Auth:Jwt:Key"])),
               ValidAudience = Configuration["Auth:Jwt:Audience"],
               ClockSkew = TimeSpan.Zero,

               // Przeczniki zwizane z bezpieczestwem
               RequireExpirationTime = true,
               ValidateIssuer = true,
               ValidateIssuerSigningKey = true,
               ValidateAudience = true
            };
         })
         // Dodanie obsugi Facebooka
         .AddFacebook(opts =>
         {
            opts.AppId = Configuration["Auth:Facebook:AppId"];
            opts.AppSecret = Configuration["Auth:Facebook:AppSecret"];
         });
      }

      // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
      public void Configure(IApplicationBuilder app, IHostingEnvironment env)
      {
         if (env.IsDevelopment())
         {
            app.UseDeveloperExceptionPage();
#if DEBUG
            app.UseWebpackDevMiddleware(new WebpackDevMiddlewareOptions
            {
               HotModuleReplacement = true
            });
#endif
         }
         else
         {
            app.UseExceptionHandler("/Home/Error");
         }

         app.UseStaticFiles(new StaticFileOptions()
         {
            OnPrepareResponse = (context) =>
            {
               // Wyczenie stosowania pamici podrcznej dla wszystkich plikw statycznych
               context.Context.Response.Headers["Cache-Control"] =
                Configuration["StaticFiles:Headers:Cache-Control"];
               context.Context.Response.Headers["Pragma"] =
                       Configuration["StaticFiles:Headers:Pragma"];
               context.Context.Response.Headers["Expires"] =
                       Configuration["StaticFiles:Headers:Expires"];
            }
         });

         // Dodaj do potoku AuthenticationMiddleware
         app.UseAuthentication();

         app.UseMvc(routes =>
         {
            routes.MapRoute(
                   name: "default",
                   template: "{controller=Home}/{action=Index}/{id?}");

            routes.MapSpaFallbackRoute(
                   name: "spa-fallback",
                   defaults: new { controller = "Home", action = "Index" });
         });

         // Utwrz zakres usugi, aby otrzyma instancj ApplicationDbContext dziki wstrzykiwaniu zalenoci
         using (var serviceScope =
         app.ApplicationServices.GetRequiredService<IServiceScopeFactory>().CreateScope())
         {
            var dbContext =
            serviceScope.ServiceProvider.GetService<ApplicationDbContext>();
            var roleManager = serviceScope.ServiceProvider.GetService<RoleManager<IdentityRole>>();
            var userManager = serviceScope.ServiceProvider.GetService<UserManager<ApplicationUser>>();
            // Utwrz baz danych, jeli nie istnieje i zastosuj wszystkie oczekujce migracje
            dbContext.Database.Migrate();
            // Wypenij baz danymi pocztkowymi
            DbSeeder.Seed(dbContext, roleManager, userManager);
         }
      }
   }
}
