/*
 * Projekt sekretnego pudeka przygotowany do Arduino 1.0 z ukadem Attiny85
 * Licencja GPL
 Wykrywa sekwencj pukania i jeeli jest poprawna, uruchamia silnik otwierajcy pudeko
 
 Steve Hoefer http://grathio.com
 Wersja 0.1.10.20.10
 Licencja Creative Commons Attribution-Noncommercial-Share Alike 3.0
 http://creativecommons.org/licenses/by-nc-sa/3.0/us/
 (W skrcie: Moesz robi z kodem, co chcesz, ale umie w nim cztery powysze wiersze i nie sprzedawaj go w adnej postaci bez porozumienia ze mn.)
 */
#include "SecretKnock.h"
#include "attinyservo.h"


SecretKnock::SecretKnock(void)
{
  // Definicje pinw
  const int programSwitch = 0;  // Jeeli stan jest wysoki, programujemy nowy wzr.
  //const int servoPin = 1;  // Pin 4 to PWM
  knockSensor = 1;         // Czujnik piezoelektryczny na pinie ADC1.
  redLED = 3;              // Status LED
  greenLED = 4;            // Status LED
  //const int speakerPin = 2;

  // Stae do tuningu. Mog to by zmienne i skojarzone z potencjometrem do regulacji, itp.
  threshold = 500;         // Minimalny poziom sygnau z czujnika piezoelektrycznego oznaczajcy pukanie
  rejectValue = 25;        // Jeeli wystukany szyfr rni si od wzorca o tyle procent, nie otwieramy pudeka.
  averageRejectValue = 15; // Jeeli redni czas pukania rni si o tyle procent, nie otwieramy pudeka.
  knockFadeTime = 200;     // Czas w milisekundach na zaniknicie puknicia przed nasuchiwaniem nastpnego (timer wytumienia)
  lockTurnTime = 650;      // Czas w milisekundach na uruchomienie silnika i wykonanie poowy obrotu
  lockMotor = 2; 
  knockComplete = 1200;     // Maksymalny czas oczekiwania, po ktrym uznajemy, e pukanie si zakoczyo


  // Zmienne
  // knockReadings[MAX_KNOCKS];   // Podczas pukania ta tabela jest wypeniana przerwami pomidzy pukniciami.
   knockSensorValue = 0;           // Ostatni odczyt z czujnika pukania.
   programButtonPressed = false;   // Ustaw flag, aby zapamita ustawienie przycisku na kocu cyklu
}

void SecretKnock::begin(int  initKnocks[])
{
  pinMode(greenLED, OUTPUT);
  pinMode(redLED, OUTPUT);
  pinMode(programSwitch, INPUT);
  
  digitalWrite(greenLED, HIGH);      // Zielona dioda LED zapalona, idziemy dalej
  delay(1000);
  digitalWrite(greenLED, LOW);      // Zielona dioda LED zgaszona, idziemy dalej
  delay(1000);
  knockReadings = initKnocks;
  servoBegin();
  moveServo(1.0, 1000);
  delay(1000);
}

void SecretKnock::checkKnock()
{
  // Nasuchuj pukania
  knockSensorValue = analogRead(knockSensor);

  //Serial.println(knockSensorValue);

  if (digitalRead(programSwitch)==HIGH){  // Czy przycisk programowania jest nacinity?
    programButtonPressed = true;          // Tak, zapisz jego stan
    digitalWrite(redLED, HIGH);           // i zapal czerwon diod, aby byo wiadomo, e trwa programowanie
  } 
  else {
    programButtonPressed = false;
    digitalWrite(redLED, LOW);
  }

  if (knockSensorValue >=threshold)
  {
    listenToSecretKnock();
  }
}

// Zapis sekwencji pukni
void SecretKnock::listenToSecretKnock()
{

  int ii = 0;
  // Najpierw zresetuj tabel do nasuchu
  for (ii=0; ii<MAX_KNOCKS; ii++){
    knockReadings[ii]=0;
  }

  int currentKnockNumber=0;          // Indeks tabeli.
  int startTime=millis();            // Odniesienie, kiedy nastpio rozpoczo si pukanie.
  int now=millis();

  digitalWrite(greenLED, LOW);       // Miganie diody, aby zwizualizowa pukanie
  if (programButtonPressed==true){
    digitalWrite(redLED, LOW);       // Zapalenie czerwonej diody, aby powiadomi o programowaniu nowego pukania
  }
  delay(knockFadeTime);              // Poczekaj na zanik sygnau, zanim zapiszesz nastpne puknicie
  digitalWrite(greenLED, HIGH);  
  if (programButtonPressed==true){
    digitalWrite(redLED, HIGH);                        
  }
  do {
    // nasuchuj nastpnego puknicia lub zaczekaj na przekroczenie czasu
    knockSensorValue = analogRead(knockSensor);
    if (knockSensorValue >=threshold){                   // nastpio nastpne puknicie...
      //record the delay time.

      now=millis();
      knockReadings[currentKnockNumber] = now-startTime;
      currentKnockNumber ++;                             // zwiksz licznik
      startTime=now;          
      // resetuj timer do nastpnego puknicia
      digitalWrite(greenLED, LOW);  
      if (programButtonPressed==true){
        digitalWrite(redLED, LOW);                       // i zapal czerwon diod, jeeli programujemy nastpne puknicie
      }
      delay(knockFadeTime);                              // znowu mae opnienie, aby ucicho puknicie
      digitalWrite(greenLED, HIGH);
      if (programButtonPressed==true){
        digitalWrite(redLED, HIGH);                         
      }
    }

    now=millis();

    // przekroczylimy czas, czy ilo pukni?
  } 
  while ((now-startTime < knockComplete) && (currentKnockNumber < MAX_KNOCKS));

  // zapisalimy puknicia, sprawdmy, czy s poprawne
  if (programButtonPressed==false)
  {  // kod wykonywany tylko wtedy, gdy nie jestemy w trybie programowania
    if (validateKnock() == true)
    {
      // triggerDoorUnlock(); 
      openDoor();
      //playSuccess();
      blinkSuccess();
    } 
    else 
    {
      digitalWrite(greenLED, LOW);          // Nie otworzylimy pudeka, mignij czerwon diod LED, aby to oznajmi
      moveServo(1.0, 100); // zamknij pudeko
      delay(1000);
      for (ii=0;ii<4;ii++){                    
        digitalWrite(redLED, HIGH);
        delay(100);
        digitalWrite(redLED, LOW);
        delay(100);
      }
      digitalWrite(greenLED, HIGH);
    }
  } 
  else { // jeeli jestemy w trybie programowania, wci sprawdzamy pukanie i nic nie robimy z zamkiem
    validateKnock();
    // migamy na zmian na zielono i czerwono, aby pokaza, e programowanie zakoczyo si
    digitalWrite(redLED, LOW);
    digitalWrite(greenLED, HIGH);
    for (ii=0;ii<3;ii++){
      delay(100);
      digitalWrite(redLED, HIGH);
      digitalWrite(greenLED, LOW);
      delay(100);
      digitalWrite(redLED, LOW);
      digitalWrite(greenLED, HIGH);      
    }
  }
}

// Uruchom silnik (lub co innego), aby otworzy pudeko
void SecretKnock::triggerDoorUnlock(){
  int ii=0;

  // uruchom na chwil silnik
  digitalWrite(lockMotor, HIGH);
  digitalWrite(greenLED, HIGH);  // i zapal zielon diod LED

  delay (lockTurnTime);          // zaczekaj chwil

  digitalWrite(lockMotor, LOW);  // zatrzymaj silnik

  // zapal kilka razy zielon diod, aby da wyrany sygna
  for (ii=0; ii < 5; ii++){   
    digitalWrite(greenLED, LOW);
    delay(100);
    digitalWrite(greenLED, HIGH);
    delay(100);
  }
}

// Funkcja sprawdzajca, czy wystukany szyfr jest zgodny ze wzorcem.
// Zwraca true, jeeli szyfr jest prawidowy, false w przeciwnym wypadku
// Do zrobienia: podzieli kod na mniejsze funkcje w celu poprawienia czytelnoci.
boolean SecretKnock::validateKnock()
{
  int ii=0;

  // Najpierw najprostszy test: czy liczba pukni si zgadza?
  int currentKnockCount = 0;
  int secretKnockCount = 0;
  int maxKnockInterval = 0;    // Tej zmiennej uyjemy pniej do normalizacji czasu.

  for (ii=0;ii<MAX_KNOCKS;ii++){
    if (knockReadings[ii] > 0){
      currentKnockCount++;
    }
    if (secretCode[ii] > 0){      // Do zrobienia: przeliczy
      secretKnockCount++;
    }

    if (knockReadings[ii] > maxKnockInterval){   // Zbieranie danych normalizacyjnych w ptli.
      maxKnockInterval = knockReadings[ii];
    }
  }

  // Jeeli programujemy nowy szyfr, zapisz informacje i wyjd std.
  if (programButtonPressed==true){
    for (ii=0;ii<MAX_KNOCKS;ii++){ // Normalizacja czasw
      secretCode[ii]= map(knockReadings[ii],0, maxKnockInterval, 0, 100); 
    }
    // Zapalaj diody zgodnie z pukaniem, aby potwierdzi jego  zaprogramowanie.
    digitalWrite(greenLED, LOW);
    digitalWrite(redLED, LOW);
    delay(1000);
    digitalWrite(greenLED, HIGH);
    digitalWrite(redLED, HIGH);
    delay(50);
    for (ii = 0; ii < MAX_KNOCKS ; ii++){
      digitalWrite(greenLED, LOW);
      digitalWrite(redLED, LOW);  
      // Zapal diod tylko wtedy, gdy jest opnienie
      if (secretCode[ii] > 0){                                   
        delay( map(secretCode[ii],0, 100, 0, maxKnockInterval)); // Rozcignij czas do mniej wicej poprzedniej wartoci
        digitalWrite(greenLED, HIGH);
        digitalWrite(redLED, HIGH);
      }
      delay(50);
    }
    return false;   // Nie otwieramy pokrywki podczas zapisywania nowego szyfru.
  }

  if (currentKnockCount != secretKnockCount){
    return false; 
  }

  /*  Teraz porwnujemy wzgldne przerwy w sekwencji pukni, a nie bezwzgldne odstpy midzy nimi
   (tzn. jeeli wystukasz t sam sekwencj szybko lub wolno, drzwi powinny zosta otwarte).
   Dziki temu ukad jest mniej wraliwy na tempo pukania, ktre moe si nieco rni od zaprogramowanego 
   wzorca, aczkolwiek obnione jest wtedy bezpieczestwo.
   */
  int totaltimeDifferences=0;
  int timeDiff=0;
  for (ii=0;ii<MAX_KNOCKS;ii++){ // Normalizacja czasw
    knockReadings[ii]= map(knockReadings[ii],0, maxKnockInterval, 0, 100);      
    timeDiff = abs(knockReadings[ii]-secretCode[ii]);
    if (timeDiff > rejectValue){ // Poszczeglne wartoci zbyt si rni
      return false;
    }
    totaltimeDifferences += timeDiff;
  }
  // Wynik jest negatywny, jeeli caa sekwencja jest zbyt niedokadna.
  if (totaltimeDifferences/secretKnockCount>averageRejectValue){
    return false; 
  }

  return true;
}

void SecretKnock::openDoor()
{
  moveServo(2.0, 100);
  delay(1000);
}

void SecretKnock::blinkSuccess()
{
  for (int ii = 0; ii < 50; ii++)
  {
    digitalWrite(greenLED, HIGH);
    delay(25);
    digitalWrite(greenLED, LOW);
    delay(25);
  }
}
