#include <LiquidCrystal.h>

/* 
Wszystkim projektom, których ten program ma nie obsługiwać należy przypisać
wartość "false". Dzięki temu brak pewnych komponentów nie będzie wywoływał
fałszywych alarmów.
*/
const boolean project4 = true; // Monitor akumulatora
const boolean project6 = true; // Alarm wyposażony w czujnik podczerwieni
const boolean project10 = true; // Czujnik otwarcia drzwi
const boolean project11 = true; // Alarm przeciwpożarowy
const boolean project12 = true; // Monitor temperatury

// Złącza, z których będziesz korzystać
const int buzzerPin = 11;
const int voltagePin = A3;
const int backlightPin = 10;
const int switchPin = A0;
const int pirPin = 2;   
const int doorPin = 12;   
const int smokePin = 3;
const int tempPin = A2;    


// Stałe projektu 2.
const float maxV = 12.6;
const float minV = 11.7;
const float warnV = 11.7;
const float R1 = 470.0;
const float R2 = 270.0;
const float k = (R1 + R2) / R2;

// Stałe projektu 12.
// wartości mogą zostać podane w stopniach Celsjusza lub w skali Fahrenheita 
const float maxTemp = 45.0;
const float minTemp = -10.0;


//                RS,E,D4,D5,D6,D7         
LiquidCrystal lcd(8, 9, 4, 5, 6, 7);

boolean mute = false;

void setup() 
{
  // Z powodu wad wielu tanich wyświetlaczy LCD,
  // podświetlenie ekranu sterowane za pomocą tranzystora i złącza D10 
  // może doprowadzić do uszkodzenia złącza płytki Arduino.
  pinMode(backlightPin, INPUT);
  pinMode(pirPin, INPUT);   
  pinMode(smokePin, INPUT);
  pinMode(doorPin, INPUT_PULLUP);
  lcd.begin(16, 2);
  Serial.begin(9600);
}

void loop() 
{
  if (Serial.available() && Serial.read() == '?')
  {
    reportStatus();
  }
  
  if (project4) checkBattery();
  if (project6) checkPIR();     
  if (project10) checkDoor(); 
  if (project11) checkSmoke();
  if (project12) checkTemp();
  
  if (analogRead(switchPin) < 1000) // wciśnięcie dowolnego przycisku
  {
    mute = ! mute;
    if (mute) 
    {
      noTone(buzzerPin);
      lcd.setCursor(12, 1);
      lcd.print("WYC.");
    }
    else
    {
      lcd.setCursor(12, 1);
      lcd.print("    ");
    }
    delay(300);
  }
  delay(100);
}

void alarm(char message[])
{
  lcd.setCursor(0, 1);
  lcd.print(message);
  delay(100);
  lcd.setCursor(0, 1);
  lcd.print("            ");
  if (!mute)
  {
    tone(buzzerPin, 1000);
  }
  delay(100);
}


void alarm(char message[], float value)
{
  alarm(message);
  lcd.setCursor(5, 1);
  lcd.print(" ");
  lcd.print(value);
}

void warn(char message[])
{
  lcd.setCursor(0, 1);
  lcd.print(message);
  delay(100);
  lcd.setCursor(0, 1);
  lcd.print("            ");
  if (!mute)
  {
    tone(buzzerPin, 1000);
    delay(100);
    noTone(buzzerPin);
  }
  delay(100);
}


void checkBattery()
{
  if (readVoltage() < warnV)
  {
    alarm("NAPIECIE");
  }
  displayVoltage();
  displayBar();
}


void displayVoltage()
{
  lcd.setCursor(0, 0);
  lcd.print("Bateria");
  lcd.setCursor(8, 0);
  lcd.print("        ");
  lcd.setCursor(8, 0);
  lcd.print(readVoltage());
  lcd.setCursor(14, 0);
  lcd.print("V");
}

float readVoltage()
{
  int raw = analogRead(voltagePin);
  float vout = (float(raw) / 1023.0) * 5.0;
  float vin = (vout * k);
  return vin;
}

void displayBar()
{
  float v = readVoltage();
  float range = maxV - minV;
  float fullness = (v - minV) / range;
  
  int numBars = fullness * 10;
  lcd.setCursor(0, 1);
  for (int i = 0; i < 10; i++)
  {
    if (numBars > i)
    {
      lcd.print("*");
    }
    else
    {
      lcd.print(" ");
    }
  }
}


void checkPIR()
{
  if (digitalRead(pirPin))
  {
    alarm("ZOMBIES!!");
  }
}

void checkDoor()
{
  if (digitalRead(doorPin))
  {
    warn("DRZWI");
  }
}

void checkSmoke()
{
  if (digitalRead(smokePin))
  {
    alarm("DYM!!!");
  }
}


void checkTemp()
{
  float t = readTemp();
  if (t > maxTemp)
  {
    alarm("GORACO", t);
  }
  else if (t < minTemp)
  {
    alarm("ZIMNO", t);
  }
}

float readTemp()
{
  int raw = analogRead(tempPin);
  float volts = raw / 205.0;
  float tempC = 100.0 * volts - 50;
  float tempF = tempC * 9.0 / 5.0 + 32.0;
  // Jedna z poniższych linii musi być oznaczona jako komentarz.
  // Temperatura może być wyrażana w stopniach Celsjusza lub w skali Fahrenheita 
  return tempC;
  // return tempF;
}

void reportStatus()
{
  Serial.print(readVoltage());
  Serial.print(" "); 
  Serial.print(readTemp());
  Serial.print(" "); 
  Serial.print(digitalRead(doorPin));
  Serial.print(" "); 
  Serial.print(digitalRead(pirPin));  
  Serial.print(" "); 
  Serial.println(digitalRead(smokePin));
}
 
