﻿using System.Globalization; // CultureInfo

SectionTitle("Specjalne wartości daty i czasu");

// optional: explicitly set culture to English (British)
/*Thread.CurrentThread.CurrentCulture = System.Globalization
  .CultureInfo.GetCultureInfo("en-GB");
*/
WriteLine($"DateTime.MinValue:  {DateTime.MinValue}");
WriteLine($"DateTime.MaxValue:  {DateTime.MaxValue}");
WriteLine($"DateTime.UnixEpoch: {DateTime.UnixEpoch}");
WriteLine($"DateTime.Now:       {DateTime.Now}");
WriteLine($"DateTime.Today:     {DateTime.Today}");

DateTime xmas = new(year: 2024, month: 12, day: 25);
WriteLine($"Boże Narodzenie (domyślny format): {xmas}");
WriteLine($"Boże Narodzenie (niestandardowy format): {xmas:dddd, dd MMMM yyyy}");
WriteLine($"Boże Narodzenie wypada w {xmas.Month}. miesiącu roku.");
WriteLine($"Boże Narodzenie wypada w {xmas.DayOfYear}. dniu 2024 r.");
WriteLine($"Dzień tygodnia, w którym wypada Boże Narodzenie w 2024 r., to {xmas.DayOfWeek}.");

SectionTitle("Działania na datach i czasie");

DateTime beforeXmas = xmas.Subtract(TimeSpan.FromDays(12));
DateTime afterXmas = xmas.AddDays(12);

// :d oznacza krótki format daty bez czasu.
WriteLine($"12 dni przed Bożym Narodzeniem: {beforeXmas:d}");
WriteLine($"12 dni po Bożym Narodzeniu: {afterXmas:d}");

TimeSpan untilXmas = xmas - DateTime.Now;

WriteLine($"Aktualna data i czas: {DateTime.Now}");
WriteLine("Do Bożego Narodzenia w 2024 r. pozostało {0} dni i {1} godzin.",
  arg0: untilXmas.Days, arg1: untilXmas.Hours);

WriteLine("Do Bożego Narodzenia w 2024 r. pozostało {0:N0} godzin.",
  arg0: untilXmas.TotalHours);

DateTime kidsWakeUp = new(
  year: 2024, month: 12, day: 25,
  hour: 6, minute: 30, second: 0);

WriteLine($"Pobudka dzieci: {kidsWakeUp}");

WriteLine("Dzieci obudzą mnie o {0}",
  arg0: kidsWakeUp.ToShortTimeString());

SectionTitle("Mili-, mikro- i nanosekundy");

DateTime preciseTime = new(
  year: 2022, month: 11, day: 8,
  hour: 12, minute: 0, second: 0,
  millisecond: 6, microsecond: 999);

WriteLine("Milisekundy: {0}, mikrosekundy: {1}, nanosekundy: {2}",
  preciseTime.Millisecond, preciseTime.Microsecond, preciseTime.Nanosecond);

preciseTime = DateTime.UtcNow;

// Liczba nanosekund zawiera się w przedziale od 0 do 900 w odstępach co 100.
WriteLine("Milisekundy: {0}, mikrosekundy: {1}, nanosekundy: {2}",
  preciseTime.Millisecond, preciseTime.Microsecond, preciseTime.Nanosecond);

SectionTitle("Globalizacja daty i czasu");

// Ustawienie takie samo jak Thread.CurrentThread.CurrentCulture.
WriteLine($"Bieżąca kultura: {CultureInfo.CurrentCulture.Name}");

string textDate = "4 lipca 2024";
DateTime independenceDay = DateTime.Parse(textDate);

WriteLine($"Tekst: {textDate}, data: {independenceDay:d MMMM}");

textDate = "7/4/2024";
independenceDay = DateTime.Parse(textDate);

WriteLine($"Tekst: {textDate}, data: {independenceDay:d MMMM}");

independenceDay = DateTime.Parse(textDate,
  provider: CultureInfo.GetCultureInfo("en-US"));

WriteLine($"Tekst: {textDate}, data: {independenceDay:d MMMM}");

for (int year = 2022; year <= 2028; year++)
{
  Write($"Rok {year} jest przestępny: {DateTime.IsLeapYear(year)}. ");
  WriteLine("Luty {0} ma {1} dni.",
    arg0: year, arg1: DateTime.DaysInMonth(year: year, month: 2));
}

WriteLine("W Boże Narodzenie obowiązuje czas letni: {0}",
  arg0: xmas.IsDaylightSavingTime());

WriteLine("W Święto Niepodległości obowiązuje czas letni: {0}",
  arg0: independenceDay.IsDaylightSavingTime());

SectionTitle("Lokalizowanie typu wyliczeniowego DayOfWeek");

CultureInfo previousCulture = Thread.CurrentThread.CurrentCulture;

// Jawne ustawienie kultury polskiej.
Thread.CurrentThread.CurrentCulture = 
  CultureInfo.GetCultureInfo("pl-PL");

WriteLine("Kultura: {0}, dzień tygodnia: {1}",
  Thread.CurrentThread.CurrentCulture.NativeName, 
  DateTime.Now.DayOfWeek);

WriteLine("Kultura: {0}, dzień tygodnia: {1:dddd}",
  Thread.CurrentThread.CurrentCulture.NativeName,
  DateTime.Now);

WriteLine("Kultura: {0}, dzień tygodnia: {1}",
  Thread.CurrentThread.CurrentCulture.NativeName,
  DateTimeFormatInfo.CurrentInfo.GetDayName(DateTime.Now.DayOfWeek));

Thread.CurrentThread.CurrentCulture = previousCulture;

SectionTitle("Działania tylko na dacie lub czasie");

DateOnly coronation = new(year: 2023, month: 5, day: 6);
WriteLine($"Dzień koronacji: {coronation.ToLongDateString()}.");

TimeOnly starts = new(hour: 11, minute: 30);
WriteLine($"Godzina rozpoczęcia: {starts}.");

DateTime calendarEntry = coronation.ToDateTime(starts);
WriteLine($"Zapis w kalendarzu: {calendarEntry}.");
