# -*- coding: UTF-8 -*-
import xlwt

from django.db import models
from django.contrib import admin
from django.utils.translation import ugettext_lazy as _
from django.http import HttpResponse

from products.models import Product, ProductPhoto


class ProductPhotoInline(admin.StackedInline):
    model = ProductPhoto
    extra = 0


def export_xls(modeladmin, request, queryset):
    response = HttpResponse(mimetype='application/ms-excel')
    response['Content-Disposition'] = 'attachment; filename=products.xls'
    wb = xlwt.Workbook(encoding='utf-8')
    ws = wb.add_sheet("Produkty")

    row_num = 0

    columns = [
        (u"ID", 2000),
        (u"Title", 6000),
        (u"Description", 8000),
        (u"Price (€)", 3000),
    ]

    header_style = xlwt.XFStyle()
    header_style.font.bold = True

    for col_num in xrange(len(columns)):
        ws.write(row_num, col_num, columns[col_num][0], header_style)
        # set column width
        ws.col(col_num).width = columns[col_num][1]

    text_style = xlwt.XFStyle()
    text_style.alignment.wrap = 1

    price_style = xlwt.XFStyle()
    price_style.num_format_str = '0.00'

    styles = [text_style, text_style, text_style, price_style]

    for obj in queryset.order_by('pk'):
        row_num += 1
        row = [
            obj.pk,
            obj.title,
            obj.description,
            obj.price,
        ]
        for col_num in xrange(len(row)):
            ws.write(row_num, col_num, row[col_num], styles[col_num])

    wb.save(response)
    return response

export_xls.short_description = u"Eksportuj do pliku XLS"


class PhotoFilter(admin.SimpleListFilter):
    # Czytelny tytuł, który zostanie wyświetlony po prawej
    # stronie panelu administracyjnego, nad opcjami filtrowania.
    title = _(u'zdjęcia')

    # Parametr dla filtru, który zostanie użyty w zapytaniu URL.
    parameter_name = 'photos'

    def lookups(self, request, model_admin):
        """
        Zwraca listę krotek. Pierwszy element w każdej
        krotce jest zakodowaną wartością opcji, która
        pojawi się w zapytaniu URL. Drugi element to
        czytelna nazwa opcji, która pojawi się na pasku
        po prawej stronie.
        """
        return (
            ('0', _(u'Nie ma zdjęć')),
            ('1', _(u'Ma jedno zdjęcie')),
            ('2+', _(u'Ma więcej niż jedno zdjęcie')),
        )

    def queryset(self, request, queryset):
        """
        Zwraca przefiltrowany zbiór obiektów na podstawie
        wartości dostarczonej w łańcuchu zapytania, który
        można pobrać przy użyciu konstrukcji self.value().
        """
        if self.value() == '0':
            return queryset.annotate(
                num_photos=models.Count('productphoto')
            ).filter(num_photos=0)
        if self.value() == '1':
            return queryset.annotate(
                num_photos=models.Count('productphoto')
            ).filter(num_photos=1)
        if self.value() == '2+':
            return queryset.annotate(
                num_photos=models.Count('productphoto')
            ).filter(num_photos__gte=2)


class ProductAdmin(admin.ModelAdmin):
    list_display = ['title', 'get_photo', 'price']
    list_editable = ['price']
    list_filter = [PhotoFilter]

    fieldsets = (
        (_('Produkt'), {'fields': ('title', 'slug', 'description', 'price')}),
    )

    prepopulated_fields = {'slug': ('title',)}
    inlines = [ProductPhotoInline]
    actions = [export_xls]

    def get_photo(self, obj):
        project_photos = obj.productphoto_set.all()[:1]
        if project_photos:
            return u"""<a href="%(product_url)s" target="_blank">
                <img src="%(photo_url)s" alt="" width="100" />
            </a>""" % {
                'product_url': obj.get_url_path(),
                'photo_url':  project_photos[0].photo.url,
            }
        return u""
    get_photo.short_description = _(u'Pierwsze zdjęcie')
    get_photo.allow_tags = True


admin.site.register(Product, ProductAdmin)
