export const UNITS = [{
  short: 't',
  milliseconds: 5 * 8 * 60 * 60 * 1000
}, {
  short: 'd',
  milliseconds: 8 * 60 * 60 * 1000
}, {
  short: 'g',
  milliseconds: 60 * 60 * 1000
}, {
  short: 'm',
  milliseconds: 60 * 1000
}];

export function formatDuration(timeSpan) {
  return UNITS.reduce((str, unit) => {
    const amount = timeSpan / unit.milliseconds;
    if (amount >= 1) {
      const fullUnits = Math.floor(amount);
      const formatted = `${str} ${fullUnits}${unit.short}`;
      timeSpan -= fullUnits * unit.milliseconds;
      return formatted;
    } else {
      return str;
    }
  }, '').trim();
}

export function parseDuration(formattedDuration) {
  const pattern = /[\d\.]+\s*[tdgm]/g;
  let timeSpan = 0;
  let result;
  while (result = pattern.exec(formattedDuration)) {
    const chunk = result[0].replace(/\s/g, '');
    let amount = Number(chunk.slice(0, -1));
    let unitShortName = chunk.slice(-1);
    timeSpan += amount * UNITS.find((unit) => unit.short === unitShortName).milliseconds;
  }
  return +timeSpan || null;
}

export function rasterize(timeData, timeFrame, quantity, now = +new Date(), fill = 0, accumulate = false) {
  // Zejdź w dół do zadanej ramki czasu.
  now = Math.floor(now / timeFrame) * timeFrame;
  // Wartość akumulowana służy do śledzenia aktualnej wartości w trybie akumulacji.
  let accumulatedValue = 0;

  // W trybie akumulacji musimy mieć pewność, że dane czasowe są posortowane.
  if (accumulate) {
    timeData = timeData.slice().sort((a, b) => a.time < b.time ? -1 : a.time > b.time ? 1 : 0);
  }

  return timeData.reduce((rasterized, timeData) => {
    // Zwiększ akumulowanąwartość, gdybyśmy jej potrzebowali.
    accumulatedValue += timeData.weight;
    // Wylicz indeks w rasteryzowanej tablicy wyjściowej.
    const index = Math.ceil((now - timeData.time) / timeFrame);
    // Jeśli indeks jest większy lub równy długości rasteryzowanej tablicy, pomiń wartość.
    if (index < quantity) {
      rasterized[index] = accumulate ? accumulatedValue : (rasterized[index] || 0) + timeData.weight;
    }
    return rasterized;
  }, Array.from({length: quantity}).fill(fill)).reverse();
}
