import {inject, async, TestBed} from '@angular/core/testing';
import {By} from '@angular/platform-browser';
import {Component, Input} from '@angular/core';
import {AutoComplete} from './auto-complete';

@Component({
  selector: 'ngc-editor',
  template: '{{content}}'
})
export class MockEditor {
  @Input() content;
  @Input() showControls;
}

describe('AutoComplete', () => {
  afterEach(() => {
    TestBed.resetTestingModule();
  });

  it('powinien poprawnie filtrować elementy', () => {
    // Dla...
    const autoComplete = new AutoComplete();
    autoComplete.items = ['jeden', 'dwa', 'trzy'];

    // Gdy...
    autoComplete.filterItems('d');

    // Wtedy...
    expect(autoComplete.filteredItems).toEqual(['dwa', 'jeden']);
  });

  it('powinien emitować zdarzenie selectedItemChange z wartością null, gdy zapis dotyczy pustej treści', () => {
    // Dla...
    const autoComplete = new AutoComplete();
    autoComplete.items = ['jeden', 'dwa', 'trzy'];
    autoComplete.selectedItem = 'trzy';
    spyOn(autoComplete.selectedItemChange, 'next');
    spyOn(autoComplete.itemCreated, 'next');

    // Gdy...
    autoComplete.onEditSaved('');

    // Wtedy...
    expect(autoComplete.selectedItemChange.next).toHaveBeenCalledWith(null);
    expect(autoComplete.itemCreated.next).not.toHaveBeenCalled();
  });

  it('powinien emitować zdarzenie itemCreated przy zapisie wartości, która nie pasuje do żadnej z już istniejących', () => {
    // Dla...
    const autoComplete = new AutoComplete();
    autoComplete.items = ['jeden', 'dwa', 'trzy'];
    autoComplete.selectedItem = 'trzy';
    spyOn(autoComplete.selectedItemChange, 'next');
    spyOn(autoComplete.itemCreated, 'next');

    // Gdy...
    autoComplete.onEditSaved('cztery');

    // Wtedy...
    expect(autoComplete.selectedItemChange.next).not.toHaveBeenCalled();
    expect(autoComplete.itemCreated.next).toHaveBeenCalledWith('cztery');
  });

  it('powinien inicjalizować edytor wybranym elementem', async(() => {
    TestBed.configureTestingModule({
      declarations: [AutoComplete, MockEditor]
    });
    TestBed.compileComponents().then(() => {
      const fixture = TestBed.createComponent(AutoComplete);
      // Dla...
      fixture.componentInstance.items = ['jeden', 'dwa', 'trzy'];
      fixture.componentInstance.selectedItem = 'dwa';

      // Gdy...
      fixture.detectChanges();

      // Wtedy...
      expect(fixture.debugElement.query(By.directive(MockEditor))
        .nativeElement.textContent.trim()).toBe('dwa');
    });
  }));

  it('powinien wyemitować zdarzenie selectedItemChange po kliknięciu w podpowiedź', async(() => {
    TestBed.configureTestingModule({
      declarations: [AutoComplete, MockEditor]
    });
    TestBed.compileComponents().then(() => {
      const fixture = TestBed.createComponent(AutoComplete);
        // Dla...
        spyOn(fixture.componentInstance.selectedItemChange, 'next');
        fixture.componentInstance.items = ['jeden', 'dwa', 'trzy'];
        fixture.componentInstance.selectedItem = 'jeden';
        fixture.componentInstance.onEditModeChange(true);
        fixture.componentInstance.onEditableInput('');

        // Gdy...
        fixture.detectChanges();
        fixture.debugElement
          .queryAll(By.css('.auto-complete__item'))
          .find((item) => item.nativeElement.textContent.trim() === 'dwa')
          .triggerEventHandler('click');

        // Wtedy...
        expect(fixture.componentInstance.selectedItemChange.next).toHaveBeenCalledWith('dwa');
      });
  }));

  it('powinien utworzyć przycisk utworzenia w podpowiedziach i wyemitować zdarzenie itemCreated po kliknięciu', async(() => {
    TestBed.configureTestingModule({
      declarations: [AutoComplete, MockEditor]
    });
    TestBed.compileComponents().then(() => {
      const fixture = TestBed.createComponent(AutoComplete);
        // Dla...
        spyOn(fixture.componentInstance.itemCreated, 'next');
        fixture.componentInstance.items = ['jeden', 'dwa', 'trzy'];
        fixture.componentInstance.selectedItem = 'jeden';
        fixture.componentInstance.onEditModeChange(true);
        fixture.componentInstance.onEditableInput('cztery');

        // Gdy...
        fixture.detectChanges();
        const createElement = fixture.debugElement
          .query(By.css('.auto-complete__item--create'));
        createElement.triggerEventHandler('click');

        // Wtedy...
        expect(createElement.nativeElement.textContent.trim()).toBe('Utwórz "cztery"');
        expect(fixture.componentInstance.itemCreated.next).toHaveBeenCalledWith('cztery');
      });
  }));
});

