import {Component, ViewEncapsulation, Input, Output, EventEmitter, ChangeDetectionStrategy, Inject} from '@angular/core';
import template from './task-list.html!text';
import {ActivityService} from '../activities/activity-service/activity-service';
import {limitWithEllipsis} from '../utilities/string-utilities';

@Component({
  selector: 'ngc-task-list',
  // Właściwość host umożliwia ustawienie niektórych właściwości
  // na elemencie HTML, który powoduje użycie komponentu.
  host: {
    class: 'task-list'
  },
  template,
  encapsulation: ViewEncapsulation.None,
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class TaskList {
  @Input() tasks;
  // Obiekt do logowania aktywności.
  @Input() activitySubject;
  // Emiter zdarzeń wykonujący zgłoszenie zdarzenia, gdy została zmieniona lista zadań.
  @Output() tasksUpdated = new EventEmitter();

  // Określ dane do filtracji.
  constructor(@Inject(ActivityService) activityService) {
    this.taskFilterList = ['wszystkie', 'otwarte', 'wykonane'];
    this.selectedTaskFilter = 'wszystkie';
    this.activityService = activityService;
  }

  ngOnChanges(changes) {
    if (changes.tasks) {
      this.taskFilterChange(this.selectedTaskFilter);
    }
  }

  // Metoda zwraca przefiltrowaną listę zadań na podstawie wybranego rodzaju filtracji.
  taskFilterChange(filter) {
    this.selectedTaskFilter = filter;
    this.filteredTasks = this.tasks ? this.tasks.filter((task) => {
      if (filter === 'wszystkie') {
        return true;
      } else if (filter === 'otwarte') {
        return !task.done;
      } else {
        return task.done;
      }
    }) : [];
  }

  // Wykorzystamy odniesienie do starego zadania, aby uaktualnić jeden konkretny element na liście zadań.
  onTaskUpdated(task, updatedData) {
    const tasks = this.tasks.slice();
    const oldTask = tasks.splice(tasks.indexOf(task), 1, Object.assign({}, task, updatedData))[0];
    tasks.splice(tasks.indexOf(task), 1, Object.assign({}, task, updatedData));
    this.tasksUpdated.next(tasks);
    // Utworzenie wpisu dotyczącego aktualizacji zadania.
    this.activityService.logActivity(
      this.activitySubject.id,
      'tasks',
      'Zaktualizowano zadanie',
      `Zaktualizowano zadanie "${limitWithEllipsis(oldTask.title, 30)}" w #${this.activitySubject.document.data._id}.`
    );
  }

  // Wykorzystamy odniesienie do zadania, funkcja usunie je z listy i wyśle uaktualnienie.
  onTaskDeleted(task) {
    const tasks = this.tasks.slice();
    const removed = tasks.splice(tasks.indexOf(task), 1)[0];
    tasks.splice(tasks.indexOf(task), 1);
    this.tasksUpdated.next(tasks);
    // Utworzenie wpisu dotyczącego usunięcia zadania.
    this.activityService.logActivity(
      this.activitySubject.id,
      'tasks',
      'Usunięto zadanie',
      `Usunięto zadanie "${limitWithEllipsis(removed.title, 30)}" z #${this.activitySubject.document.data._id}.`
    );
  }

  // Funkcja dodająca zadanie z widoku.
  addTask(title) {
    const tasks = this.tasks.slice();
    tasks.push({
      position: tasks.length,
      created: +new Date(),
      title,
      done: null
    });
    this.tasksUpdated.next(tasks);
    // Utworzenie wpisu dotyczącego dodania zadania.
    this.activityService.logActivity(
      this.activitySubject.id,
      'tasks',
      'Dodano nowe zadanie',
      `Dodano nowe zadanie "${limitWithEllipsis(title, 30)}" do #${this.activitySubject.document.data._id}.`
    );
  }

  onTaskDrop(source, target) {
    if (source.position === target.position) {
      return;
    }

    let tasks = this.tasks.slice();
    const sourceIndex = tasks.findIndex(
      (task) => task.position === source.position
    );
    const targetIndex = tasks.findIndex(
      (task) => task.position === target.position
    );
    tasks.splice(targetIndex, 0, tasks.splice(sourceIndex, 1)[0]);
    tasks = tasks.map((task, index) => {
      return Object.assign({}, task, {
        position: index
      });
    });
    this.tasksUpdated.next(tasks);

    this.activityService.logActivity(
      this.activitySubject.id,
      'tasks',
      'Przeniesiono zadanie',
      `Zadanie "${limitWithEllipsis(source.title, 30)}" z #${this.activitySubject.document.data._id} przeniesiono na pozycję ${target.position + 1}.`
    );
  }
}
