package com.manning.aip.portfolio;

import java.util.ArrayList;

import android.app.ListActivity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.View.OnClickListener;
import android.widget.BaseAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import com.manning.aip.portfolio.service.IStockService;

/**
 * Gwna aktywno aplikacji. Udostpnia prosty formularz do wprowadzania
 * danych o akcjach, ktre uytkownik chce ledzi, a take wywietla list 
 * obserwowanych akcji. Aktywno korzysta z usugi 
 * {@link com.manning.aip.portfolio.PortfolioManagerService PortfolioManagerService}
 * do pobierania portfela akcji i ich dodawania.
 * 
 * @author Michael Galpin
 *
 */
public class ViewStocks extends ListActivity {

	private static final String LOGGING_TAG = "ViewStocks";
	
	// Lista akcji wywietlanych uytkownikowi.
	private ArrayList<Stock> stocks;
	// Usuga uywana do utrwalania i pobierania akcji.
	private IStockService stockService;
	// Czy usuga jest obecnie podczona?
	private boolean bound = false;
	
	// Poczenie z usug i kod do obsugi zdarze zwizanych z cyklem ycia usugi.
	private ServiceConnection connection = new ServiceConnection(){

		public void onServiceConnected(ComponentName className, 
				IBinder service) {
			stockService = IStockService.Stub.asInterface(service);
			Log.d(LOGGING_TAG,"Podczono usug");
			try {
				stocks = (ArrayList<Stock>) stockService.getPortfolio();
				if (stocks == null){
					stocks = new ArrayList<Stock>(0);
					Log.d(LOGGING_TAG, "Usuga nie zwrci akcji");
				} else {
					Log.d(LOGGING_TAG, "Pobrano "+ stocks.size() +" akcji z usugi");
				}
				refresh();
			} catch (RemoteException e) {
				Log.e(LOGGING_TAG, "Wyjtek przy pobieraniu portfela akcji z usugi",e);
			}
		}

		public void onServiceDisconnected(ComponentName className) {
			stockService = null;
			Log.d(LOGGING_TAG,"Zerwano poczenie z usug");
		}
		
	};
	
	@Override
	public void onStart(){
		super.onStart();
        // Tworzenie wyjciowej listy.
		if (!bound){
			bound = bindService(
					new Intent(ViewStocks.this, PortfolioManagerService.class), 
					connection, Context.BIND_AUTO_CREATE);
			Log.d(LOGGING_TAG, "Podczono usug: " + bound);
		}
		if (!bound){
			Log.e(LOGGING_TAG, "Nieudane podczenie usugi");
			throw new RuntimeException("Nieudane wyszukiwanie usugi");
		}
		setListAdapter(new BaseAdapter(){

			public int getCount() {
				if (stocks == null){
					return 0;
				}
				return stocks.size();
			}

			public Object getItem(int position) {
				if (stocks == null){
					return null;
				}
				return stocks.get(position);
			}

			public long getItemId(int position) {
				if (stocks == null){
					return 0L;
				}
				return stocks.get(position).getId();
			}

			public View getView(int position, View convertView, 
					ViewGroup parent) {
				if (convertView == null){
					LayoutInflater inflater = getLayoutInflater();
					convertView = 
						inflater.inflate(R.layout.stock, parent, false);
				}
				TextView rowTxt = 
					(TextView) convertView.findViewById(R.id.rowTxt);
				rowTxt.setText(stocks.get(position).toString());
				return convertView;
			}

			@Override
			public boolean hasStableIds() {
				return true;
			}
        	
        });
	}
	
	@Override
	public void onPause(){
		super.onPause();
		if (bound){
			bound = false;
			unbindService(connection);
		}
	}
	
    @Override
    public void onCreate(Bundle savedInstanceState) {
    	// Tworzenie elementw interfejsu uytkownika.
    	// Dane zostay wczytane w metodzie <code>onStart</code>.
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);    

        // Dodawanie kontrolek.
        final EditText symbolIn = (EditText) findViewById(R.id.inputSymbol);
        final EditText maxIn = (EditText) findViewById(R.id.inputMax);
        final EditText minIn = (EditText) findViewById(R.id.inputMin);
        final EditText priceIn = (EditText) findViewById(R.id.inputPrice);
        final EditText quantIn = (EditText) findViewById(R.id.inputQuant);
        
        // Dodawanie procedury obsugi kliknicia przycisku.
		Button button = (Button) findViewById(R.id.btn);
		button.setOnClickListener(new OnClickListener(){
			public void onClick(View v) {
				String symbol = symbolIn.getText().toString();
				symbolIn.setText("");
				double max = Double.parseDouble(maxIn.getText().toString());
				maxIn.setText("");
				double min = Double.parseDouble(minIn.getText().toString());
				minIn.setText("");
				double pricePaid = 
					Double.parseDouble(priceIn.getText().toString());
				priceIn.setText("");
				int quantity = Integer.parseInt(quantIn.getText().toString());
				quantIn.setText("");
				Stock stock = new Stock(symbol, pricePaid, quantity);
				stock.setMaxPrice(max);
				stock.setMinPrice(min);
				// Dodawanie akcji do portfela za pomoc dziaajcej w tle usugi.
				new AsyncTask<Stock,Void,Stock>(){
					@Override
					protected Stock doInBackground(Stock... newStocks) {						
						try {
							return stockService.addToPortfolio(newStocks[0]);
						} catch (RemoteException e) {
							Log.e(LOGGING_TAG, "Wyjtek w czasie dodawania " +
									"akcji do portfela", e);
						}
						return null;
					}
					@Override
					protected void onPostExecute(Stock s){
						Log.d(LOGGING_TAG, "Akcje zwrcone przez usug: " + s);
						if (s == null){
							Log.w(LOGGING_TAG, "Usuga zwrcia null " +
									"lub nieprawidowe dane");
							Toast.makeText(ViewStocks.this, "Akcji nie znaleziono", 
									Toast.LENGTH_SHORT);
						} else { 
							refreshStockData();
						}
					}
				}.execute(stock);
			}
		});
    }

	@Override
	protected void onDestroy() {
		super.onDestroy();
		// Zerwanie poczenia z usug.
		unbindService(connection);
	}

    // Aktualizowanie danych o akcjach otrzymanych od usugi
	// i odwieanie interfejsu uytkownika.
	private void refreshStockData() {
		if (stocks != null && stocks.size() > 0){
			new AsyncTask<Void, Void, ArrayList<Stock>>(){
				@Override
				protected void onPostExecute(ArrayList<Stock> result) {
					Log.d(LOGGING_TAG, "Otrzymano nowe dane z usugi");
					if (result != null){
						stocks = result;
						refresh();
					} else {
						Toast.makeText(ViewStocks.this, "Wyjtek przy pobieraniu " +
								"najnowszych danych o akcjach", Toast.LENGTH_SHORT);
					}
				}
	
				@Override
				protected ArrayList<Stock> doInBackground(Void... nada){
					try {
						return (ArrayList<Stock>) stockService.getPortfolio();
					} catch (Exception e) {
						Log.e(LOGGING_TAG, "Wyjtek przy pobieraniu danych o akcjach", e);
					}
					return null;
				}
	    	}.execute();
		}
	}
    
    private void refresh(){
    	Log.d(LOGGING_TAG, "Odwieanie interfejsu uytkownika na podstawie nowych danych");
		for (Stock s : stocks){
			Log.d(LOGGING_TAG, "Otrzymano dane o akcjach: " + s.toString());
		}
    	BaseAdapter adapter = (BaseAdapter) this.getListAdapter();
    	adapter.notifyDataSetChanged();
    }
}