// DetailFragment.java
// Podklasa Fragment wyświetlająca szczegóły jednego kontaktu.
package com.deitel.addressbook;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.support.v4.app.DialogFragment;
import android.support.v4.app.Fragment;
import android.support.v4.app.LoaderManager;
import android.support.v4.content.CursorLoader;
import android.support.v4.content.Loader;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import com.deitel.addressbook.data.DatabaseDescription.Contact;

public class DetailFragment extends Fragment
   implements LoaderManager.LoaderCallbacks<Cursor> {

   // metody wywołania zwrotnego implementowane przez aktywność MainActivity
   public interface DetailFragmentListener {
      void onContactDeleted(); // metoda wywoływana w przypadku kasowania kontaktu

      // przekaż adres Uri kontaktu, który ma być edytowany, do interfejsu DetailFragmentListener
      void onEditContact(Uri contactUri);
   }

   private static final int CONTACT_LOADER = 0; // identyfikuje obiekt Loader

   private DetailFragmentListener listener; // MainActivity
   private Uri contactUri; // adres Uri wybranego kontaktu

   private TextView nameTextView; // wyświetla imię przypisane do wybranego kontaktu
   private TextView phoneTextView; // wyświetla numer telefonu przypisany do wybranego kontaktu
   private TextView emailTextView; // wyświetla adres poczty elektronicznej przypisany do wybranego kontaktu
   private TextView streetTextView; // wyświetla nazwę ulicy przypisaną do wybranego kontaktu
   private TextView cityTextView; // wyświetla nazwę miasta przypisaną do wybranego kontaktu
   private TextView stateTextView; // wyświetla nazwę państwa przypisaną do wybranego kontaktu
   private TextView zipTextView; // wyświetla kod pocztowy przypisany do wybranego kontaktu

   // inicjuj interfejs DetailFragmentListener, gdy fragment zostanie dołączony
   @Override
   public void onAttach(Context context) {
      super.onAttach(context);
      listener = (DetailFragmentListener) context;
   }

   // usuń interfejs DetailFragmentListener, gdy fragment zostanie odłączony
   @Override
   public void onDetach() {
      super.onDetach();
      listener = null;
   }

   // metoda wywoływana, gdy musi zostać utworzony widok interfejsu DetailFragmentListener
   @Override
   public View onCreateView(
      LayoutInflater inflater, ViewGroup container,
      Bundle savedInstanceState) {
      super.onCreateView(inflater, container, savedInstanceState);
      setHasOptionsMenu(true); // fragment ten posiada elementy menu, które muszą zostać wyświetlone

      // uzyskuje obiekt Bundle zawierający argumenty, a następnie odczytuje z nich adres Uri kontaktu
      Bundle arguments = getArguments();

      if (arguments != null)
         contactUri = arguments.getParcelable(MainActivity.CONTACT_URI);

      // przygotuj do wyświetlenia rozkład fragmentu DetailFragment
      View view =
         inflater.inflate(R.layout.fragment_detail, container, false);

      // uzyskaj pola EditText
      nameTextView = (TextView) view.findViewById(R.id.nameTextView);
      phoneTextView = (TextView) view.findViewById(R.id.phoneTextView);
      emailTextView = (TextView) view.findViewById(R.id.emailTextView);
      streetTextView = (TextView) view.findViewById(R.id.streetTextView);
      cityTextView = (TextView) view.findViewById(R.id.cityTextView);
      stateTextView = (TextView) view.findViewById(R.id.stateTextView);
      zipTextView = (TextView) view.findViewById(R.id.zipTextView);

      // załaduj kontakt
      getLoaderManager().initLoader(CONTACT_LOADER, null, this);
      return view;
   }

   // wyświetla elementy menu tego fragmentu
   @Override
   public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
      super.onCreateOptionsMenu(menu, inflater);
      inflater.inflate(R.menu.fragment_details_menu, menu);
   }

   // obsłuż wybranie przez użytkownika elementu menu
   @Override
   public boolean onOptionsItemSelected(MenuItem item) {
      switch (item.getItemId()) {
         case R.id.action_edit:
            listener.onEditContact(contactUri); // przekaż adres Uri do obiektu nasłuchującego
            return true;
         case R.id.action_delete:
            deleteContact();
            return true;
      }

      return super.onOptionsItemSelected(item);
   }

   // kasuj kontakt
   private void deleteContact() {
      // skorzystaj z menedżera FragmentManager w celu wyświetlenia fragmentu DialogFragment confirmDelete
      confirmDelete.show(getFragmentManager(), "confirm delete");
   }

   // fragment DialogFragment służący do potwierdzania chęci skasowania kontaktu
   private final DialogFragment confirmDelete =
      new DialogFragment() {
         // utwórz i zwróć okno AlertDialog
         @Override
         public Dialog onCreateDialog(Bundle bundle) {
            // utwórz nowy obiekt AlertDialog Builder
            AlertDialog.Builder builder =
               new AlertDialog.Builder(getActivity());

            builder.setTitle(R.string.confirm_title);
            builder.setMessage(R.string.confirm_message);

            // utwórz przycisk OK zamykający okno
            builder.setPositiveButton(R.string.button_delete,
               new DialogInterface.OnClickListener() {
                  @Override
                  public void onClick(
                     DialogInterface dialog, int button) {

                     // skorzystaj z obiektu ContentResolver tej aktywności w celu wywołania
                     // metody delete obiektu AddressBookContentProvider
                     getActivity().getContentResolver().delete(
                        contactUri, null, null);
                     listener.onContactDeleted(); // powiadom obiekt nasłuchujący zdarzeń
                  }
               }
            );

            builder.setNegativeButton(R.string.button_cancel, null);
            return builder.create(); // zwróć okno AlertDialog
         }
      };

   // metoda wywoływana przez menedżera LoaderManager w celu utworzenia obiektu Loader
   @Override
   public Loader<Cursor> onCreateLoader(int id, Bundle args) {
      // na podstawie argumentu określającego identyfikator utwórz właściwy obiekt CursorLoader;
      //  w tym fragmencie znajduje się tylko jeden obiekt Loader, a więc instrukcja switch jest zbędna
      CursorLoader cursorLoader;

      switch (id) {
         case CONTACT_LOADER:
            cursorLoader = new CursorLoader(getActivity(),
               contactUri, // adres Uri kontaktu, który ma zostać wyświetlony
               null, // rzutowanie wartości null zwraca wszystkie kolumny
               null, // wybranie wartości null zwraca wszystkie rzędy
               null, // brak argumentów selekcji
               null); // kolejność sortowania
            break;
         default:
            cursorLoader = null;
            break;
      }

      return cursorLoader;
   }

   // metoda wywoływana przez menedżera LoaderManager po zakończeniu ładowania danych
   @Override
   public void onLoadFinished(Loader<Cursor> loader, Cursor data) {
      // jeżeli kontakt istnieje w bazie danych, to wyświetl jego dane
      if (data != null && data.moveToFirst()) {
         // odczytaj indeks kolumny dla każdego elementu danych kontaktu
         int nameIndex = data.getColumnIndex(Contact.COLUMN_NAME);
         int phoneIndex = data.getColumnIndex(Contact.COLUMN_PHONE);
         int emailIndex = data.getColumnIndex(Contact.COLUMN_EMAIL);
         int streetIndex = data.getColumnIndex(Contact.COLUMN_STREET);
         int cityIndex = data.getColumnIndex(Contact.COLUMN_CITY);
         int stateIndex = data.getColumnIndex(Contact.COLUMN_STATE);
         int zipIndex = data.getColumnIndex(Contact.COLUMN_ZIP);

         // odczytane dane wyświetl w polach TextView
         nameTextView.setText(data.getString(nameIndex));
         phoneTextView.setText(data.getString(phoneIndex));
         emailTextView.setText(data.getString(emailIndex));
         streetTextView.setText(data.getString(streetIndex));
         cityTextView.setText(data.getString(cityIndex));
         stateTextView.setText(data.getString(stateIndex));
         zipTextView.setText(data.getString(zipIndex));
      }
   }

   // metoda wywoływana przez menedżera LoaderManager gdy obiekt Loader jest wyzerowywany
   @Override
   public void onLoaderReset(Loader<Cursor> loader) { }
}


/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
