// WeatherArrayAdapter.java
// Klasa ArrayAdapter służy do wyświetlania elementów obiektu List<Weather> w widoku ListView.
package com.deitel.weatherviewer;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.AsyncTask;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.ImageView;
import android.widget.TextView;

import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

class WeatherArrayAdapter extends ArrayAdapter<Weather> {
   // klasa służąca do ponownego używania widoków usuwanych z ekranu w wyniku przewijania elementów listy
   private static class ViewHolder {
      ImageView conditionImageView;
      TextView dayTextView;
      TextView lowTextView;
      TextView hiTextView;
      TextView humidityTextView;
   }

   // przechowuje pobrane bitmapy, umożliwiając ich ponowne użycie
   private Map<String, Bitmap> bitmaps = new HashMap<>();

   // konstruktor służący do inicjalizacji składowych odziedziczonych z klasy nadrzędnej
   public WeatherArrayAdapter(Context context, List<Weather> forecast) {
      super(context, -1, forecast);
   }

   // tworzy zmodyfikowane widoki dla elementów listy ListView
   @Override
   public View getView(int position, View convertView, ViewGroup parent) {
      // uzyskaj dostęp do obiektu Weather właściwego dla danego punktu listy ListView
      Weather day = getItem(position);

      ViewHolder viewHolder; // obiekt, który odwołuje się do widoków elementów listy

      // sprawdź, czy istnieje nadający się do ponownego użycia obiekt ViewHolder przypisany do elementu
      // listy ListView usuniętego z ekranu; jeżeli obiekt taki nie istnieje, to utwórz nowy obiekt ViewHolder
      if (convertView == null) { // brak obiektu ViewHolder nadającego się do ponownego użycia; utwórz nowy obiekt ViewHolder
         viewHolder = new ViewHolder();
         LayoutInflater inflater = LayoutInflater.from(getContext());
         convertView =
            inflater.inflate(R.layout.list_item, parent, false);
         viewHolder.conditionImageView =
            (ImageView) convertView.findViewById(R.id.conditionImageView);
         viewHolder.dayTextView =
            (TextView) convertView.findViewById(R.id.dayTextView);
         viewHolder.lowTextView =
            (TextView) convertView.findViewById(R.id.lowTextView);
         viewHolder.hiTextView =
            (TextView) convertView.findViewById(R.id.hiTextView);
         viewHolder.humidityTextView =
            (TextView) convertView.findViewById(R.id.humidityTextView);
         convertView.setTag(viewHolder);
      }
      else { // skorzystaj jeszcze raz z istniejącego obiektu ViewHolder przechowywanego jako znacznik elementu listy
         viewHolder = (ViewHolder) convertView.getTag();
      }

      // jeżeli pobrano wcześniej ikonę ilustrującą daną pogodę, to skorzystaj z niej;
      // w przeciwnym wypadku pobierz ikonę za pomocą oddzielnego wątku
      if (bitmaps.containsKey(day.iconURL)) {
         viewHolder.conditionImageView.setImageBitmap(
            bitmaps.get(day.iconURL));
      }
      else {
         // pobierz i wyświetl obraz ilustrujący warunki pogodowe
         new LoadImageTask(viewHolder.conditionImageView).execute(
            day.iconURL);
      }

      // odczytaj pozostałe dane z obiektu Weather i umieść je w odpowiednich widokach
      Context context = getContext(); // do ładowania zasobów będących łańcuchami znaków
      viewHolder.dayTextView.setText(context.getString(
         R.string.day_description, day.dayOfWeek, day.description));
      viewHolder.lowTextView.setText(
         context.getString(R.string.low_temp, day.minTemp));
      viewHolder.hiTextView.setText(
         context.getString(R.string.high_temp, day.maxTemp));
      viewHolder.humidityTextView.setText(
         context.getString(R.string.humidity, day.humidity));

      return convertView; // zwróć kompletną listę elementów do wyświetlenia
   }

   // Podklasa AsyncTask ładująca za pomocą oddzielnego wątku ikony ilustrujące warunki atmosferyczne
   private class LoadImageTask extends AsyncTask<String, Void, Bitmap> {
      private ImageView imageView; // wyświetla miniaturę

      // przechowaj obiekt ImageView, na którym ma zostać wyświetlona pobrana bitmapa
      public LoadImageTask(ImageView imageView) {
         this.imageView = imageView;
      }

      // załaduj obraz; params[0] łańcuchem znaków definiującym adres URL obrazu
      @Override
      protected Bitmap doInBackground(String... params) {
         Bitmap bitmap = null;
         HttpURLConnection connection = null;

         try {
            URL url = new URL(params[0]); // utwórz obiekt URL obrazu

            // otwórz połączenie HttpURLConnection, uzyskaj dostęp do strumienia InputStream
            // i pobierz obraz
            connection = (HttpURLConnection) url.openConnection();

            try (InputStream inputStream = connection.getInputStream()) {
               bitmap = BitmapFactory.decodeStream(inputStream);
               bitmaps.put(params[0], bitmap); // zapisz w pamięci podręcznej
            }
            catch (Exception e) {
               e.printStackTrace();
            }
         }
         catch (Exception e) {
            e.printStackTrace();
         }
         finally {
            connection.disconnect(); // zamknij połączenie HttpURLConnection
         }

         return bitmap;
      }

      // dołącz obraz do elementu listy
      @Override
      protected void onPostExecute(Bitmap bitmap) {
         imageView.setImageBitmap(bitmap);
      }
   }
}

/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
