// MainActivity.java
// Zawiera obiekt MainActivityFragment po uruchomieniu aplikacji na telefonie lub obiekty
// MainActivityFragment i SettingsActivityFragment po uruchomieniu aplikacji na tablecie.
package com.deitel.flagquiz;

import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.OnSharedPreferenceChangeListener;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.Toast;

import java.util.Set;

public class MainActivity extends AppCompatActivity {
   // klucze umożliwiające wczytanie danych z SharedPreferences
   public static final String CHOICES = "pref_numberOfChoices";
   public static final String REGIONS = "pref_regionsToInclude";

   private boolean phoneDevice = true; // wymusza tryb portretowy
   private boolean preferencesChanged = true; // Czy preferencje zostały zmienione?

   // konfiguruje MainActivity
   @Override
   protected void onCreate(Bundle savedInstanceState) {
      super.onCreate(savedInstanceState);
      setContentView(R.layout.activity_main);
      Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
      setSupportActionBar(toolbar);

      // przypisuje domyślne wartości do SharedPreferences
      PreferenceManager.setDefaultValues(this, R.xml.preferences, false);

      // rejestruje obiekt nasłuchujący zmian SharedPreferences
      PreferenceManager.getDefaultSharedPreferences(this).
         registerOnSharedPreferenceChangeListener(
            preferencesChangeListener);

      // określa rozmiar ekranu
      int screenSize = getResources().getConfiguration().screenLayout &
         Configuration.SCREENLAYOUT_SIZE_MASK;

      // jeżeli urządzenie jest tabletem, przypisuje wartość false zmiennej phoneDevice
      if (screenSize == Configuration.SCREENLAYOUT_SIZE_LARGE ||
         screenSize == Configuration.SCREENLAYOUT_SIZE_XLARGE)
         phoneDevice = false; // urządzenie o wymiarach innych niż telefon

      // jeżeli aplikacja działa na urządzeniu mającym wymiary telefonu, to zezwalaj tylko na orientację pionową
      if (phoneDevice)
         setRequestedOrientation(
            ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
   }

   // wywoływana po skończeniu wykonywania metody onCreate
   @Override
   protected void onStart() {
      super.onStart();

      if (preferencesChanged) {
         // Teraz gdy domyślne preferencje zostały ustawione,
         // zainicjuj MainActivityFragment i uruchom quiz.
         MainActivityFragment quizFragment = (MainActivityFragment)
            getSupportFragmentManager().findFragmentById(
               R.id.quizFragment);
         quizFragment.updateGuessRows(
            PreferenceManager.getDefaultSharedPreferences(this));
         quizFragment.updateRegions(
            PreferenceManager.getDefaultSharedPreferences(this));
         quizFragment.resetQuiz();
         preferencesChanged = false;
      }
   }

   // pokaż menu, jeżeli aplikację uruchomiono na telefonie lub tablecie w orientacji pionowej
   @Override
   public boolean onCreateOptionsMenu(Menu menu) {
      // ustal aktualną orientację urządzenia
      int orientation = getResources().getConfiguration().orientation;

      // wyświetla menu aplikacji tylko w orientacji pionowej
      if (orientation == Configuration.ORIENTATION_PORTRAIT) {
         // wygeneruj menu
         getMenuInflater().inflate(R.menu.menu_main, menu);
         return true;
      }
      else
         return false;
   }

   // wyświetla SettingsActivity, gdy aplikacja jest uruchomiona na telefonie
   @Override
   public boolean onOptionsItemSelected(MenuItem item) {
      Intent preferencesIntent = new Intent(this, SettingsActivity.class);
      startActivity(preferencesIntent);
      return super.onOptionsItemSelected(item);
   }

   // nasłuchuje zmian obiektu SharedPreferences
   private OnSharedPreferenceChangeListener preferencesChangeListener =
      new OnSharedPreferenceChangeListener() {
         // wywoływane, gdy użytkownik zmienia preferencje aplikacji
         @Override
         public void onSharedPreferenceChanged(
            SharedPreferences sharedPreferences, String key) {
            preferencesChanged = true; // użytkownik zmienił ustawienia aplikacji

            MainActivityFragment quizFragment = (MainActivityFragment)
               getSupportFragmentManager().findFragmentById(
                  R.id.quizFragment);

            if (key.equals(CHOICES)) { // zmiana liczby wyświetlanych odpowiedzi
               quizFragment.updateGuessRows(sharedPreferences);
               quizFragment.resetQuiz();
            }
            else if (key.equals(REGIONS)) { // zmiana obszarów, których ma dotyczyć quiz
               Set<String> regions =
                  sharedPreferences.getStringSet(REGIONS, null);

               if (regions != null && regions.size() > 0) {
                  quizFragment.updateRegions(sharedPreferences);
                  quizFragment.resetQuiz();
               }
               else {
                  // przynajmniej jeden obszar musi zostać wybrany - domyślnie jest to Ameryka Północna
                  SharedPreferences.Editor editor =
                     sharedPreferences.edit();
                  regions.add(getString(R.string.default_region));
                  editor.putStringSet(REGIONS, regions);
                  editor.apply();

                  Toast.makeText(MainActivity.this,
                     R.string.default_region_message,
                     Toast.LENGTH_SHORT).show();
               }
            }

            Toast.makeText(MainActivity.this,
               R.string.restarting_quiz,
               Toast.LENGTH_SHORT).show();
         }
      };
}


/*************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 *************************************************************************/
