// MainActivity.java
// Klasa obsługująca komunikację pomiędzy obiektami Fragment, które się w niej znajdują.
package com.deitel.addressbook;

import android.net.Uri;
import android.os.Bundle;
import android.support.v4.app.FragmentTransaction;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;

public class MainActivity extends AppCompatActivity
   implements ContactsFragment.ContactsFragmentListener,
   DetailFragment.DetailFragmentListener,
   AddEditFragment.AddEditFragmentListener {

   // klucz przeznaczony do przechowywania adresu Uri kontaktu w obiekcie przekazywanym do fragmentu
   public static final String CONTACT_URI = "contact_uri";

   private ContactsFragment contactsFragment; // wyświetla listę kontaktów

   // wyświetl fragment ContactsFragment, gdy rozkład MainActivity jest ładowany jako pierwszy
   @Override
   protected void onCreate(Bundle savedInstanceState) {
      super.onCreate(savedInstanceState);
      setContentView(R.layout.activity_main);
      Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
      setSupportActionBar(toolbar);

      // jeżeli rozkład zawiera fragment fragmentContainer, oznacza to używanie rozkładu przeznaczonego dla telefonu
      // utwórz i wyświetl fragment ContactsFragment
      if (savedInstanceState == null &&
         findViewById(R.id.fragmentContainer) != null) {
         // utwórz fragment ContactsFragment
         contactsFragment = new ContactsFragment();

         // dodaj fragment do rozkładu FrameLayout
         FragmentTransaction transaction =
            getSupportFragmentManager().beginTransaction();
         transaction.add(R.id.fragmentContainer, contactsFragment);
         transaction.commit(); // wyświetl obiekt ContactsFragment
      }
      else {
         contactsFragment =
            (ContactsFragment) getSupportFragmentManager().
               findFragmentById(R.id.contactsFragment);
      }
   }

   // wyświetl fragment DetailFragment dla wybranego kontaktu
   @Override
   public void onContactSelected(Uri contactUri) {
      if (findViewById(R.id.fragmentContainer) != null) // telefon
         displayContact(contactUri, R.id.fragmentContainer);
      else { // tablet
         // usuwa górny element ze stosu programowego
         getSupportFragmentManager().popBackStack();

         displayContact(contactUri, R.id.rightPaneContainer);
      }
   }

   // wyświetl fragment AddEditFragment dodający nowy kontakt
   @Override
   public void onAddContact() {
      if (findViewById(R.id.fragmentContainer) != null) // telefon
         displayAddEditFragment(R.id.fragmentContainer, null);
      else // tablet
         displayAddEditFragment(R.id.rightPaneContainer, null);
   }

   // wyświetl kontakt
   private void displayContact(Uri contactUri, int viewID) {
      DetailFragment detailFragment = new DetailFragment();

      // przekaż adres Uri kontaktu jako argument fragmentu DetailFragment
      Bundle arguments = new Bundle();
      arguments.putParcelable(CONTACT_URI, contactUri);
      detailFragment.setArguments(arguments);

      // skorzystaj z transakcji FragmentTransaction w celu wyświetlenia fragmentu DetailFragment
      FragmentTransaction transaction =
         getSupportFragmentManager().beginTransaction();
      transaction.replace(viewID, detailFragment);
      transaction.addToBackStack(null);
      transaction.commit(); // powoduje wyświetlenie fragmentu DetailFragment
   }

   // wyświetl fragment umożliwiający dodanie nowego kontaktu lub edycję zapisanego wcześniej kontaktu
   private void displayAddEditFragment(int viewID, Uri contactUri) {
      AddEditFragment addEditFragment = new AddEditFragment();

      // w przypadku edycji zapisanego wcześniej kontaktu jako argument przekaż contactUri
      if (contactUri != null) {
         Bundle arguments = new Bundle();
         arguments.putParcelable(CONTACT_URI, contactUri);
         addEditFragment.setArguments(arguments);
      }

      // skorzystaj z transakcji FragmentTransaction w celu wyświetlenia fragmentu AddEditFragment
      FragmentTransaction transaction =
         getSupportFragmentManager().beginTransaction();
      transaction.replace(viewID, addEditFragment);
      transaction.addToBackStack(null);
      transaction.commit(); // powoduje wyświetlenie fragmentu AddEditFragment
   }

   // po skasowaniu kontaktu wróć do listy kontaktów
   @Override
   public void onContactDeleted() {
      // usuwa element znajdujący się na szczycie stosu aplikacji
      getSupportFragmentManager().popBackStack();
      contactsFragment.updateContactList(); // odśwież listę kontaktów
   }

   // wyświetl fragment AddEditFragment w celu edytowania zapisanego wcześniej kontaktu
   @Override
   public void onEditContact(Uri contactUri) {
      if (findViewById(R.id.fragmentContainer) != null) // telefon
         displayAddEditFragment(R.id.fragmentContainer, contactUri);
      else // tablet
         displayAddEditFragment(R.id.rightPaneContainer, contactUri);
   }

   // zaktualizuj graficzny interfejs użytkownika po dodaniu nowego kontaktu lub zaktualizowaniu zapisanego wcześniej kontaktu
   @Override
   public void onAddEditCompleted(Uri contactUri) {
      // zdejmuje obiekt znajdujący się na górze stosu
      getSupportFragmentManager().popBackStack();
      contactsFragment.updateContactList(); // odśwież kontakty

      if (findViewById(R.id.fragmentContainer) == null) { // tablet
         // usuń górny element stosu
         getSupportFragmentManager().popBackStack();

         // w aplikacji uruchomionej na tablecie wyświetl kontakt, który został dodany lub zmodyfikowany
         displayContact(contactUri, R.id.rightPaneContainer);
      }
   }
}


/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
