package com.androidbook.salbcr;

import android.content.Context;
import android.os.PowerManager;
import android.util.Log;

/*
 * Czym jest zielony pokj?
 * ****************************************************
 * Zielony pokj jest pomieszczeniem, do ktrego mog wchodzi gocie.
 * Pocztkowo wiata w tym pokoju s pogaszone.
 * Pierwszy go, ktry "wejdzie", zapala wiato.
 * Kolejni gocie nie maj wpywu na stan owietlenia pokoju, gdy wiato
 * zostao wczone.
 * Ostatni go w momencie wyjcia zgasi wiato.
 * Metody s zsynchronizowane w taki sposb, aby utrzymywa stan.
 * "Wejcia" i "wyjcia" mog wystpowa pomidzy wieloma wtkami.
 * Nazwa "zielony pokj" wzia si od moliwoci oszczdzania energii.
 * Licha definicja, ale chyba dobrze wyjania omawiane koncepcje.
 * 
 * Czym jest owietlony, zielony pokj?
 * ****************************************************
 * W przeciwiestwie do zielonego pokoju, w ktrym pocztkowo wiata s pogaszone,
 * owietlony, zielony pokj ma od razu wczone wiata.
 * Ostatni go zgasi wiato w momencie wyjcia.
 * 
 * Rozwizanie to przydaje si w przypadku obecnoci dwch punktw wejciowych. 
 * Jeeli jeden z tych punktw jest opniony, 
 * to pomieszczenie niezawierajce listy umwionych goci
 * moe samo wygasi wiato.
 * 
 * Odbieramy kontrol gociowi "wchodzcemu" do
 * pokoju, gdy jest ju on owietlony.
 * Stan "wej" jest obserwowany tylko po to, aby wiedzie, czy
 * ostatni go ju wyszed.
 * 
 */
public class LightedGreenRoom 
{
	//znacznik debugowania
	private static String tag="LightedGreenRoom";
	
	//*************************************************
	//* Publiczny interfejs statyczny
	//*   statyczni czonkowie: same metody pomocniczne
	//*   Deleguje do obiektu singletowego
	//*************************************************
	public static void setup(Context inCtx)
	{
		if (s_self == null)
		{
			Log.d(LightedGreenRoom.tag,"Tworzenie i oswietlanie zielonego pokoju");
			s_self = new LightedGreenRoom(inCtx);
			s_self.turnOnLights();
		}
	}
	public static boolean isSetup(){
		return (s_self != null) ? true: false; 
	}
	public static int s_enter() {
		assertSetup();
		return s_self.enter();
	}
	public static int s_leave(){
		assertSetup();
		return s_self.leave();
	}
	//Nie wywoujmy bezporednio tej metody,
	//gdy stanie si prawdopodobnie przestarzaa.
	//Zamiast tego wywoujmy klienckie metody rejestrujce i wyrejestrujce.
	public static void ds_emptyTheRoom(){
		assertSetup();
		s_self.emptyTheRoom();
		return;
	}
	public static void s_registerClient(){
		assertSetup();
		s_self.registerClient();
		return;
	}
	public static void s_unRegisterClient(){
		assertSetup();
		s_self.unRegisterClient();
		return;
	}
	private static void assertSetup(){
		if (LightedGreenRoom.s_self == null)
		{
			Log.w(LightedGreenRoom.tag,"Musimy najpierw wywolac konfigurator");
			throw new RuntimeException("Musimy najpierw skonfigurowac klase GreenRoom");
		}
	}

	//*************************************************
	//* Cakowicie prywatna implementacja
	//*************************************************
	
	//Zlicza ilo goci, aby wiedzie, ktry jest ostatni.
	//Podczas zamykania usugi zeruje licznik w celu wyczyszczenia pokoju.
	private int count;
	
	//Potrzebna do utworzenia blokady budzenia
	private Context ctx = null;
	
	//Nasz przecznik
	PowerManager.WakeLock wl = null;
	
	//Obsuga wielu klientw
	private int clientCount = 0;
	
	/*
	* Oczekujemy, e bdzie to klasa singletonowa.
	* Potencjalnie mona zrobi prywatnego
	* konstruktora.
	*/

	private LightedGreenRoom(Context inCtx)
	{
		ctx = inCtx;
		wl = this.createWakeLock(inCtx);
	}
	
	/*
	* Konfigurowanie zielonego pokoju za pomoc statycznej metody.
	* Musi by wywoana przed wywoaniem innych metod.
	* jej zadania:
	* 1. Tworzenie wystpienia obiektu
	* 2. Wprowadzenie blokady wczajcej wiata
	* Zaoenie:
	* Nie musi by synchronizowana,
	* poniewa bdzie wywoywana z gwnego wtku.
	* (Moe by bdne. Naley to sprawdzi!!)
	*/
	private static LightedGreenRoom s_self = null;
	
	/*
	* Spodziewamy si, e metody "wchodzenia" i "wychodzenia"
	* bd wsplnie wywoywane.
	*
	* Przy "wejciu" licznik jest zwikszany.
	*
	* Nie wczamy ani nie wyczamy wiate,
	* poniewa s one ju zapalone.
	*
	* Zwikszamy warto licznika tylko po to,
    * aby wiedzie, kiedy wyjdzie ostatni go.
	*
	* Jest to synchronizowana metoda, poniewa
	* bdzie wchodzio i wychodzio wiele wtkw.
	*
	*/
	synchronized private int enter()
	{
		count++;
		Log.d(tag,"Nowy gosc: licznik:" + count);
		return count;
	}
	/*
	* Spodziewamy si, e metody "wchodzenia" i "wychodzenia"
	* bd wsplnie wywoywane.
	*
	* Przy "wyjciu" zmniejszamy licznik.
	*
	* Jeeli licznik osignie warto 0, gasimy wiata.
	*
	* Jest to metoda synchroniczna, poniewa
	* wiele wtkw bdzie wchodzio i wychodzio.
	*
	*/
	synchronized private int leave()
	{
		Log.d(tag,"Opuszczanie pokoju: liczenie podczas wywolania:" + count);
		//Jeeli warto licznika ju wynosi zero
	    //po prostu wychodzimy.
		if (count == 0) 
		{
			Log.w(tag,"Wartosc licznika wynosi 0.");
			return count;
		}
		count--;
		if (count == 0)
		{
	        //Ostatni go
	        //gasi wiata
			turnOffLights();
		}
		return count;
	}
	/*
	 * Obecnie nikt nie stosuje tej metody.
	 * Umieszczamy j na wszelki wypadek.
	 */
	synchronized private int getCount()
	{
		return count;
	}
	
	/*
	* wprowadzamy blokad budzenia, aby zapali wiata
	* Od innych synchronizowanych metod zaley, czy zostanie ona
	* wywoana we waciwym momencie.
	*/
	private void turnOnLights()
	{
		Log.d(tag, "Wlaczanie swiatel: Licznik:" + count);
		this.wl.acquire();
	}
	
	/*
	 * Zwalnia blokad budzenia, aby zgasi wiata.
     * Od innych synchronizowanych metod zaley, czy zostanie ona
     * wywoana we waciwym momencie.
	 */
	private void turnOffLights()
	{
		if (this.wl.isHeld())
		{
			Log.d(tag,"Zwalnianie blokady budzenia. Brak gosci.");
			this.wl.release();
		}
	}
	/*
	 * Standardowy kod, sucy do utworzenia czciowej blokady budzenia.
	 */
	private PowerManager.WakeLock createWakeLock(Context inCtx)
	{
		PowerManager pm = 
			(PowerManager)inCtx.getSystemService(Context.POWER_SERVICE); 

		PowerManager.WakeLock wl = pm.newWakeLock
		     (PowerManager.PARTIAL_WAKE_LOCK, tag);
		return wl;
	}
	
	private int registerClient()
	{
		Utils.logThreadSignature(tag);
		this.clientCount++;
		Log.d(tag,"rejestrowanie nowego klienta: licznik:" + clientCount);
		return clientCount;
	}
	
	private int unRegisterClient()
	{
		Utils.logThreadSignature(tag);
		Log.d(tag,"wyrejestrowanie nowego klienta: licznik:" + clientCount);
		if (clientCount == 0)
		{
			Log.w(tag,"Brak klientow do wyrejestrowania.");
			return 0;
		}
		//Warto clientCount nie jest rwna 0
		clientCount--;
		if (clientCount == 0)
		{
			emptyTheRoom();
		}
		return clientCount;
	}
	synchronized private void emptyTheRoom()
	{
		Log.d(tag, "Wywolywana w celu wyczyszczenia pokoju");
		count = 0;
		this.turnOffLights();
	}
}
