#!/usr/bin/perl

# huff_dekodowanie( $drzewo, \&pobierz_bit, \&zwroc_strumien )
#    Akceptuje strumien bitow (dostarczanych pojedynczo przez &$pobierz_bit).
#    Konwertuje symbole wedlug kodowania Huffman definiowanego przez $drzewo,
#    i zwraca odkodowane dane wykorzystujac &$zwroc_strumien.
#
sub huff_dekodowanie {
    my $drzewo       = shift;
    my $pobierz_bit    = shift;
    my $zwroc_strumien = shift;

    my $biezacy_wezel = $drzewo;
    my $biezacy_bit;

    while( defined( $biezacy_bit = &$pobierz_bit ) ) {
        # W dol do nastepnego wezla.
        $biezacy_wezel = $biezacy_bit ? $biezacy_wezel->{right} : $biezacy_wezel->{left};
        unless( $biezacy_wezel->{left} ) {
            # W koncowym wezle - zwracamy wartosc i zaczynamy od poczatku.
            &$zwroc_strumien( $biezacy_wezel->{value} );
            $biezacy_wezel = $drzewo;
        }
    }
}

# huff_hash_poddrzewo( $wezel, \%hash, $przedrostek )
#    Zapelnia tablice asocjacyjna %hash. Schodzi na dol drzewa
#    do koncowych wezlow, rekurencyjnie zaglebiajac sie w prawe,
#    lub lewe poddrzewo, budujac po drodze @$przedrostek.
#    Kazdy klucz zawarty %hash bedzie jedna z wartosci wystepujacych
#    w oryginalnym pliku. A odpowiadajaca mu wartosc, to bity
#    ktore ten klucz koduja.
#    $przedrostek to sekwencja bitow definiujaca $wezel.
sub huff_hash_poddrzewo {
    my $wezel   = shift;
    my $hash   = shift;
    my $przedrostek = shift;

    if( $wezel->{left} ) {
        huff_hash_poddrzewo( $wezel->{left}, $hash, [ @$przedrostek, 0 ] );
        huff_hash_poddrzewo( $wezel->{right}, $hash, [ @$przedrostek, 1 ] );
    } else {
        $hash->{$wezel->{value}} = $przedrostek;
    }
}

# huff_hash( $drzewo, \%hash )
#    Zapelniamy %hash. Uzywamy funkcji huff_hash_poddrzewo(), 
#    zaczynajac z bazowego wierzcholka drzewa.
sub huff_hash {
    my $drzewo = shift;
    my $hash = shift;
    %$hash = ( );

    huff_hash_poddrzewo( $drzewo, $hash, [] );
}

# huff_kodowanie( $hash, \&pobierz_strumien, \&zwroc_bit )
#    Dla danego strumienia symboli, wczytujemy je po jednym za pomoca $pobierz_strumien,
#    konwertujemy do spakowanego formatu kodowania Huffmana zdefiniowanego przez
#    funkcje huff_hash(), i zwracamy wynik uzywajac $zwroc_bit.
sub huff_kodowanie {
    my $hash       = shift;
    my $pobierz_strumien = shift;
    my $zwroc_bit    = shift;

    my $biezacy_strumien;
    my $biezace_bity;

    while( defined( $biezacy_strumien = &$pobierz_strumien ) ) {
        # Konwertujemy na bity kodu ASCII.
        foreach $biezacy_bit (@{ $hash->{$biezacy_strumien} }) {
            # Koncowy wezel - zwracamy wartosc i zaczynamy od poczatku.
            &$zwroc_bit( $biezacy_bit );
        }
    }
}

# $drzewo = zbuduj_hash_z_drzewa( \%wagi )
#    Konwertujemy tablice asocjacyjna symboli
#    Na drzewo dekodujace Huffmana.
sub zbuduj_hash_z_drzewa {
    my $hash = shift;
    my $lista = [ ];
    my( $symbol, $waga );

    # Tworzymy koncowy wezel dla kazdego symbolu.
    while( ($symbol, $waga) = each(%$hash) ) {
        push( @$lista, {
                value => $symbol,
                weight => $waga,
            } );
    }

    # Redukujemy liste do pojedynczego drzewa.
    while( $#$lista ) {
        @$lista = sort
            { $a->{weight} <=> $b->{weight} ||
              $a->{value}  <=> $b->{value}     }
            @$lista;
        my( $lewy, $prawy ) = splice( @$lista, 0, 2 );
        my $nowy_wezel = {
                left   => $lewy,
                right  => $prawy,
                weight => $lewy->{weight} + $prawy->{weight},
                value  => $lewy->{value} . $prawy->{value},
            };
        unshift( @$lista, $nowy_wezel );
    }

    # Drzewo bedzie tym, elementem listy, ktory zostanie na koniec.
    return $lista->[0];
}

%wagi_symboli = (
    'a',  5,
    'b',  2,
    'c',  9,
    'd',  1,
    'e',  1,
    'f',  12,
);


