import csv
import random
import math

# Algorytm mrówkowy
# Algorytm mrówkowy jest oparty na zachowaniu mrówek, które w trakcie poruszania się między punktami pozostawiają 
# feromony i uwzględniają feromony innych osobników. Wynikające z tego zachowanie polega na konwergentnym wybieraniu 
# ścieżek, które wymagają najmniej wysiłku.

# Określanie liczby atrakcji w zbiorze danych.
# Najlepsza łączna odległość 5 atrakcji: 19
# Najlepsza łączna odległość 48 atrakcji: 33523
ATTRACTION_COUNT = 48
# Inicjowanie dwuwymiarowej macierzy do przechowywania odległości między atrakcjami.
attraction_distances = []
# Wczytywanie zbioru danych z odległościami między atrakcjami i zapisywanie go w macierzy.
with open('attractions-' + str(ATTRACTION_COUNT) + '.csv') as file:
    reader = csv.reader(file, quoting=csv.QUOTE_NONNUMERIC)
    for row in reader:
        attraction_distances.append(row)

# Ustawianie prawdopodobieństwa wyboru losowej atrakcji przez mrówki (0.0 - 1.0).
RANDOM_ATTRACTION_FACTOR = 0.0
# Waga feromonów przy wyborze ścieżek.
ALPHA = 4
# Waga heurystyki przy wyborze ścieżek.
BETA = 7

# Klasa Ant reprezentuje mrówkę z algorytmu mrówkowego.
# Mrówki poruszają się między różnymi atrakcjami i pozostawiają feromony. Wybierają też atrakcję, którą mają
# odwiedzić w następnej kolejności. Znają też łączną przebytą przez siebie odległość.
# - Pamięć: w algorytmie mrówkowym jest to lista już odwiedzonych atrakcji.
# - Najlepsze przystosowanie: najmniejsza odległość na drodze między wszystkimi atrakcjami.
# - Działanie: wybór następnej lokalizacji do odwiedzenia i pozostawianie feromonów na drodze.
class Ant:

    # Mrówka początkowo nie odwiedziła żadnych atrakcji i losowo wybierana jest jedna z atrakcji.
    def __init__(self):
        self.visited_attractions = []
        self.visited_attractions.append(random.randint(0, ATTRACTION_COUNT - 1))
        
    # Wybieranie atrakcji losowo lub za pomocą algorytmu.
    def visit_attraction(self, pheromone_trails):
        if random.random() < RANDOM_ATTRACTION_FACTOR:
            self.visited_attractions.append(self.visit_random_attraction())
        else:
            self.visited_attractions.append(
                self.roulette_wheel_selection(self.visit_probabilistic_attraction(pheromone_trails)))

    # Losowe wybieranie atrakcji.
    def visit_random_attraction(self):
        all_attractions = set(range(0, ATTRACTION_COUNT))
        possible_attractions = all_attractions - set(self.visited_attractions)
        return random.randint(0, len(possible_attractions) - 1)

    # Obliczanie prawdopodobieństw odwiedzin przyległych nieodwiedzonych atrakcji.
    def visit_probabilistic_attraction(self, pheromone_trails):
        current_attraction = self.visited_attractions[-1]
        all_attractions = set(range(0, ATTRACTION_COUNT))
        possible_attractions = all_attractions - set(self.visited_attractions)
        possible_indexes = []
        possible_probabilities = []
        total_probabilities = 0
        for attraction in possible_attractions:
            possible_indexes.append(attraction)
            pheromones_on_path = math.pow(pheromone_trails[current_attraction][attraction], ALPHA)
            heuristic_for_path = math.pow(1 / attraction_distances[current_attraction][attraction], BETA)
            probability = pheromones_on_path * heuristic_for_path
            possible_probabilities.append(probability)
            total_probabilities += probability
        possible_probabilities = [probability / total_probabilities for probability in possible_probabilities]
        return [possible_indexes, possible_probabilities, len(possible_attractions)]

    # Wybieranie atrakcji na podstawie prawdopodobieństw odwiedzin przyległych nieodwiedzonych atrakcji.
    @staticmethod
    def roulette_wheel_selection(probabilities):
        slices = []
        total = 0
        possible_indexes = probabilities[0]
        possible_probabilities = probabilities[1]
        possible_attractions_count = probabilities[2]
        for i in range(0, possible_attractions_count):
            slices.append([possible_indexes[i], total, total + possible_probabilities[i]])
            total += possible_probabilities[i]
        spin = random.random()
        result = [s[0] for s in slices if s[1] < spin <= s[2]]
        return result[0]

    # Pobieranie łącznej odległości przebytej przez daną mrówkę.
    def get_distance_travelled(self):
        total_distance = 0
        for a in range(1, len(self.visited_attractions)):
            total_distance += attraction_distances[self.visited_attractions[a]][self.visited_attractions[a-1]]
        total_distance += attraction_distances[self.visited_attractions[0]][self.visited_attractions[len(self.visited_attractions) - 1]]
        return total_distance

    def print_info(self):
        print('Mrówka ', self.__hash__())
        print('Atrakcji w sumie: ', len(self.visited_attractions))
        print('Łączna odległość: ', self.get_distance_travelled())


# Klasa ACO zawiera funkcje algorytmu mrówkowego.
# Oto ogólny cykl życia algorytmu mrówkowego:

# - Inicjowanie śladów feromonowych: tworzenie śladów feromonowych między atrakcjami i inicjowanie
# poziomu ich intensywności.

# - Tworzenie populacji mrówek: tworzenie populacji mrówek, gdzie każda mrówka rozpoczyna trasę 
# od innej atrakcji.

# - Wybór następnej atrakcji dla każdej mrówki: wybieranie kolejnej atrakcji do odwiedzenia przez każdą mrówkę. 
# Proces ten powtarza się do czasu jednokrotnego odwiedzenia przez każdą mrówkę każdej atrakcji.

# - Aktualizowanie śladów feromonowych: aktualizowanie intensywności śladów feromonowych na podstawie ruchów
# mrówek oraz szybkości wyparowywania feromonów.

# - Aktualizowanie najlepszego rozwiązania: aktualizowanie najlepszego rozwiązania na podstawie
# łącznej odległości przebytej przez każdą mrówkę.

# - Określanie warunku zakończenia: odwiedzanie atrakcji przez mrówki trwa określoną liczbę iteracji. Jedna
# iteracja odpowiada jednokrotnemu odwiedzeniu wszystkich atrakcji przez każdą mrówkę. Warunek zakończenia to łączna 
# liczba wykonywanych iteracji. Większa liczba iteracji pozwala mrówkom podejmować lepsze decyzje na podstawie śladów
# feromonowych.
class ACO:

    def __init__(self, number_of_ants_factor):
        self.number_of_ants_factor = number_of_ants_factor
        # Inicjowanie tablicy przechowującej mrówki.
        self.ant_colony = []
        # Inicjowanie dwuwymiarowej macierzy ze śladami feromonowymi.
        self.pheromone_trails = []
        # Inicjowanie najlepszej odległości w grupie.
        self.best_distance = math.inf
        self.best_ant = None

    # Inicjowanie mrówek zaczynających drogę od losowych lokalizacji.
    def setup_ants(self, number_of_ants_factor):
        number_of_ants = round(ATTRACTION_COUNT * number_of_ants_factor)
        self.ant_colony.clear()
        for i in range(0, number_of_ants):
            self.ant_colony.append(Ant())

    # Inicjowanie śladów feromonowych między atrakcjami.
    def setup_pheromones(self):
        for r in range(0, len(attraction_distances)):
            pheromone_list = []
            for i in range(0, len(attraction_distances)):
                pheromone_list.append(1)
            self.pheromone_trails.append(pheromone_list)

    # Przenoszenie wszystkich mrówek do nowych atrakcji.
    def move_ants(self, ant_population):
        for ant in ant_population:
            ant.visit_attraction(self.pheromone_trails)

    # Określanie najlepszej mrówki w kolonii po jednokrotnych odwiedzinach wszystkich atrakcji.
    def get_best(self, ant_population):
        for ant in ant_population:
            distance_travelled = ant.get_distance_travelled()
            if distance_travelled < self.best_distance:
                self.best_distance = distance_travelled
                self.best_ant = ant
        return self.best_ant

    # Aktualizowanie śladów feromonowych na podstawie ruchów mrówek po jednokrotnych odwiedzinach wszystkich atrakcji.
    def update_pheromones(self, evaporation_rate):
        for x in range(0, ATTRACTION_COUNT):
            for y in range(0, ATTRACTION_COUNT):
                self.pheromone_trails[x][y] = self.pheromone_trails[x][y] * evaporation_rate
                for ant in self.ant_colony:
                    self.pheromone_trails[x][y] += 1 / ant.get_distance_travelled()

    # Łączenie wszystkich elementów (główna pętla).
    def solve(self, total_iterations, evaporation_rate):
        self.setup_pheromones()
        for i in range(0, TOTAL_ITERATIONS):
            self.setup_ants(NUMBER_OF_ANTS_FACTOR)
            for r in range(0, ATTRACTION_COUNT - 1):
                self.move_ants(self.ant_colony)
            self.update_pheromones(evaporation_rate)
            self.best_ant = self.get_best(self.ant_colony)
            print(i, ' Najlepsza odległość: ', self.best_ant.get_distance_travelled())


# Ustawianie procenta mrówek na podstawie łącznej liczby atrakcji.
NUMBER_OF_ANTS_FACTOR = 0.5
# Określanie liczby tras, jakie mrówki muszą pokonać.
TOTAL_ITERATIONS = 10000
# Ustawianie współczynnika wyparowywania feromonów (0.0 - 1.0).
EVAPORATION_RATE = 0.4
aco = ACO(NUMBER_OF_ANTS_FACTOR)
aco.solve(TOTAL_ITERATIONS, EVAPORATION_RATE)
