/*************************************************************************
 *  Kompilacja:  javac EdgeWeightedDigraph.java
 *  Wykonanie:    java EdgeWeightedDigraph V E
 *  Zalenoci: Bag.java DirectedEdge.java
 *
 *  Digraf waony, zaimplementowany za pomoc list ssiedztwa.
 *
 *************************************************************************/

/**
 *  Klasa <tt>EdgeWeightedDigraph</tt> reprezentuje graf skierowany z wierzchokami
 *  o nazwach od 0 do V-1, gdzie kada krawd ma wag w postaci liczby rzeczywistej.
 *  Obsuguje nastpujce operacje: dodawanie krawdzi do grafu,
 *  przechodzenie po wszystkich krawdziach wychodzcych z wierzchoka.
 *  Krawdzie rwnolege i ptle zwrotne s dozwolone.
 *  <p>
 *  Dodatkow dokumentacj znajdziesz w <a href="http://algs4.cs.princeton.edu/44sp">podrozdziale 4.4</a> ksiki
 *  <i>Algorytmy, wydanie czwarte</i> Roberta Sedgewicka i Kevina Wayne'a.
 */



public class EdgeWeightedDigraph {
    private final int V;
    private int E;
    private Bag<DirectedEdge>[] adj;
    
    /**
     * Tworzy pusty digraf waony o V wierzchokach.
     */
    public EdgeWeightedDigraph(int V) {
        if (V < 0) throw new RuntimeException("Liczba wierzchokw musi by nieujemna");
        this.V = V;
        this.E = 0;
        adj = (Bag<DirectedEdge>[]) new Bag[V];
        for (int v = 0; v < V; v++)
            adj[v] = new Bag<DirectedEdge>();
    }

   /**
     * Tworzy digraf waony o V wierzchokach i E krawdziach.
     */
    public EdgeWeightedDigraph(int V, int E) {
        this(V);
        if (E < 0) throw new RuntimeException("Liczba wierzchokw musi by nieujemna");
        for (int i = 0; i < E; i++) {
            int v = (int) (Math.random() * V);
            int w = (int) (Math.random() * V);
            double weight = Math.round(100 * Math.random()) / 100.0;
            DirectedEdge e = new DirectedEdge(v, w, weight);
            addEdge(e);
        }
    }

    /**
     * Tworzy digraf waony na podstawie strumienia wejciowego.
     */
    public EdgeWeightedDigraph(In in) {
        this(in.readInt());
        int E = in.readInt();
        for (int i = 0; i < E; i++) {
            int v = in.readInt();
            int w = in.readInt();
            double weight = in.readDouble();
            addEdge(new DirectedEdge(v, w, weight));
        }
    }


   /**
     * Zwraca liczb wierzchokw w digrafie.
     */
    public int V() {
        return V;
    }

   /**
     * Zwraca liczb krawdzi w digrafie.
     */
    public int E() {
        return E;
    }


   /**
     * Dodaje krawd e do digrafu.
     */
    public void addEdge(DirectedEdge e) {
        int v = e.from();
        adj[v].add(e);
        E++;
    }


   /**
     * Zwraca wychodzce z wierzchoka v krawdzie jako obiekt Iterable.
     * Aby przej po krawdziach wychodzcych z wierzchoka v, uyj zapisu foreach:
     * <tt>for (DirectedEdge e : graph.adj(v))</tt>.
     */
    public Iterable<DirectedEdge> adj(int v) {
        return adj[v];
    }

   /**
     * Zwraca wszystkie krawdzie grafu jako obiekt Iterable.
     * Aby przej po krawdziach, uyj zapisu foreach:
     * <tt>for (DirectedEdge e : graph.edges())</tt>.
     */
    public Iterable<DirectedEdge> edges() {
        Bag<DirectedEdge> list = new Bag<DirectedEdge>();
        for (int v = 0; v < V; v++) {
            for (DirectedEdge e : adj(v)) {
                list.add(e);
            }
        }
        return list;
    } 

   /**
     * Zwraca liczb krawdzi wychodzcych z v.
     */
    public int outdegree(int v) {
        return adj[v].size();
    }



   /**
     * Zwraca reprezentacj acucha znakw dla grafu.
     */
    public String toString() {
        String NEWLINE = System.getProperty("line.separator");
        StringBuilder s = new StringBuilder();
        s.append(V + " " + E + NEWLINE);
        for (int v = 0; v < V; v++) {
            s.append(v + ": ");
            for (DirectedEdge e : adj[v]) {
                s.append(e + "  ");
            }
            s.append(NEWLINE);
        }
        return s.toString();
    }

    /**
     * Klient testowy.
     */
    public static void main(String[] args) {
        int V = Integer.parseInt(args[0]);
        int E = Integer.parseInt(args[1]);
        EdgeWeightedDigraph G = new EdgeWeightedDigraph(V, E);
        StdOut.println(G);
    }

}
