/*************************************************************************
 *  Kompilacja:  javac EdgeWeightedGraph.java
 *  Wykonanie:    java EdgeWeightedGraph V E
 *  Zalenoci: Bag.java Edge.java
 *
 *  Nieskierowany graf waony zaimplementowany za pomoc list ssiedztwa.
 *  Krawdzie rwnolege i ptle zwrotne s dozwolone
 *
 *************************************************************************/

/**
 *  Klasa <tt>EdgeWeightedGraph</tt> reprezentuje nieskierowany graf wierzchokw o nazwach
 *  od 0 do V-1, gdzie kada krawd ma wag o wartoci rzeczywistej.
 *  Obsuguje nastpujce operacje: dodawanie krawdzi do grafu,
 *  przechodzenie po wszystkich ssiadach wierzchoka.
 *  Krawdzie rwnolege i ptle zwrotne s dozwolone.
 *  <p>
 *  Dodatkow dokumentacj mona znale w <a href="http://algs4.cs.princeton.edu/43mst">podrozdziale 4.3</a> ksiki
 *  <i>Algorytmy, wydanie czwarte</i> Roberta Sedgewicka i Kevina Wayne'a.
 */


public class EdgeWeightedGraph {
    private final int V;
    private int E;
    private Bag<Edge>[] adj;
    
   /**
     * Tworzenie pustego grafu waonego o V wierzchokach.
     */
    public EdgeWeightedGraph(int V) {
        if (V < 0) throw new RuntimeException("Liczba wierzcholkow musi byc nieujemna");
        this.V = V;
        this.E = 0;
        adj = (Bag<Edge>[]) new Bag[V];
        for (int v = 0; v < V; v++) adj[v] = new Bag<Edge>();
    }

   /**
     * Tworzenie losowego grafu waonego o V wierzchokach i E krawdziach.
     * Oczekiwany czas wykonania jest proporcjonalny do V + E.
     */
    public EdgeWeightedGraph(int V, int E) {
        this(V);
        if (E < 0) throw new RuntimeException("Liczba krawedzi musi byc nieujemna");
        for (int i = 0; i < E; i++) {
            int v = (int) (Math.random() * V);
            int w = (int) (Math.random() * V);
            double weight = Math.round(100 * Math.random()) / 100.0;
            Edge e = new Edge(v, w, weight);
            addEdge(e);
        }
    }

   /**
     * Tworzenie grafu waonego na podstawie strumienia wejciowego.
     */
    public EdgeWeightedGraph(In in) {
        this(in.readInt());
        int E = in.readInt();
        for (int i = 0; i < E; i++) {
            int v = in.readInt();
            int w = in.readInt();
            double weight = in.readDouble();
            Edge e = new Edge(v, w, weight);
            addEdge(e);
        }
    }

   /**
     * Zwraca liczb wierzchokw w grafie.
     */
    public int V() {
        return V;
    }

   /**
     * Zwracanie liczby krawdzi w grafie.
     */
    public int E() {
        return E;
    }


   /**
     * Dodawanie krawdzi e do grafu.
     */
    public void addEdge(Edge e) {
        int v = e.either();
        int w = e.other(v);
        adj[v].add(e);
        adj[w].add(e);
        E++;
    }


   /**
     * Zwracanie krawdzi incydentnych do wierzchoka v jako obiektu Iterable.
     * Aby przej po krawdziach incydentnych do wierzchoka v naley uy zapisu foreach:
     * <tt>for (Edge e : graph.adj(v))</tt>.
     */
    public Iterable<Edge> adj(int v) {
        return adj[v];
    }

   /**
     * Zwracanie wszystkich krawdzi grafu jako obiektu Iterable.
     * Aby przej po krawdziach, naley uy zapisu foreach:
     * <tt>for (Edge e : graph.edges())</tt>.
     */
    public Iterable<Edge> edges() {
        Bag<Edge> list = new Bag<Edge>();
        for (int v = 0; v < V; v++) {
            int selfLoops = 0;
            for (Edge e : adj(v)) {
                if (e.other(v) > v) {
                    list.add(e);
                }
                // Dodawanie tylko jednej kopii kadej ptli zwrotnej
                else if (e.other(v) == v) {
                    if (selfLoops % 2 == 0) list.add(e);
                    selfLoops++;
                }
            }
        }
        return list;
    }



   /**
     * Zwracanie acucha znakw reprezentujcego dany graf
     */
    public String toString() {
        String NEWLINE = System.getProperty("line.separator");
        StringBuilder s = new StringBuilder();
        s.append(V + " " + E + NEWLINE);
        for (int v = 0; v < V; v++) {
            s.append(v + ": ");
            for (Edge e : adj[v]) {
                s.append(e + "  ");
            }
            s.append(NEWLINE);
        }
        return s.toString();
    }

   /**
     * Klient testowy
     */
    public static void main(String[] args) {
        EdgeWeightedGraph G;

        if (args.length == 0) {
            // Wczytywanie grafu ze standardowego wejcia
            G = new EdgeWeightedGraph(new In());
        }
        else {
            // Losowy graf o V wierzchokach i E krawdziach z dozwolonymi krawdziami rwnolegymi
            int V = Integer.parseInt(args[0]);
            int E = Integer.parseInt(args[1]);
            G = new EdgeWeightedGraph(V, E);
        }

        StdOut.println(G);
    }

}
