/// <remark>
/// Klasa GeneratePrimes generuje liczby pierwsze od 1 do wartoci
/// maksymalnej okrelonej przez uytkownika. Ponisza klasa
/// wykorzystuje klasyczny algorytm sita Eratostenesa.
///
/// Eratostenes (ur. 276 p.n.e. w Cyrenie, zm. 194 p.n.e. w Aleksandrii)
///  pierwszy czowiek, ktremu udao si oszacowa rednic Ziemi.
/// Eratostenes jest te znany ze swoich prac nad kalendarzami z latami
/// przestpnymi i z przejcia po Kallimachu z Cyreny zarzdzania 
/// Bibliotek Aleksandryjsk.
///
/// Algorytm sita Eratostenesa jest do prosty. Pocztkowo dysponujemy
/// tablic kolejnych liczb cakowitych (poczwszy do 2). Usuwamy z tej
/// tablicy wszystkie wielokrotnoci dwjki i przechodzimy do kolejnej nieusunitej
/// liczby cakowitej, by usun jej wielokrotnoci. Powtarzamy t procedur
/// a do osignicia wartoci rwnej pierwiastkowi kwadratowemu liczby
/// maksymalnej.
///
/// Oryginalna wersja tego kodu jest dzieem Roberta C. Martina i powstaa
/// 9 grudnia 1999. Kod Javy zosta przeoony na jzyk C# 12 stycznia 2005
/// roku przez Micaha Martina.
///</remark>

using System;

/// <summary>
/// autor: Robert C. Martin
/// </summary>
public class GeneratePrimes
{
  ///<summary>
  /// Generuje tablic liczb pierwszych.
  ///</summary>
  ///
  /// <param name="maxValue">Warto maksymalna generatora.</param>
  public static int[] GeneratePrimeNumbers(int maxValue)
  {
    if (maxValue >= 2) // jedyny prawidowy przypadek
    {
      // deklaracje
      int s = maxValue + 1; // rozmiar tablicy
      bool[] f = new bool[s];
      int i;

      // Inicjalizuje tablic wartoci true.
      for (i = 0; i < s; i++)
        f[i] = true;

      // Eliminuje znane liczby niepierwsze.
      f[0] = f[1] = false;

      // sito
      int j;
      for (i = 2; i < Math.Sqrt(s) + 1; i++)
      {
        if(f[i]) // Jeli i nie zostao wyeliminowane, eliminujemy jego wielokrotnoci.
        {
          for (j = 2 * i; j < s; j += i)
            f[j] = false; // Wielokrotno i nie moe by liczb pierwsz.
        }
      }

      // Ile liczb pierwszych zostao w naszej tablicy?
      int count = 0;
      for (i = 0; i < s; i++)
      {
        if (f[i])
          count++; // zwikszajcy si licznik
      }

      int[] primes = new int[count];

      // Przenosimy liczby pierwsze do tablicy wynikowej.
      for (i = 0, j = 0; i < s; i++)
      {
        if (f[i])             // Jeli i jest liczb pierwsz...
          primes[j++] = i;
      }

      return primes;  // Zwraca liczby pierwsze.
    }
    else // maxValue < 2
      return new int[0]; // W razie nieprawidowych danych wejciowych zwraca tablic pust.
  }
}
